//----------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Node.js Proxy Layer.
//
// Usage guide for the LoggingFoot FTSDK in a Node.js Proxy Layer.
//
// Copyright © 2025 Quantyworks Software
//----------------------------------------------------------------------------------------------
#include "ftsdk_proxy.h"

using namespace Napi;

//----------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Node.js Proxy Layer.
//
// Node-addon-api Create instance.
//----------------------------------------------------------------------------------------------
Napi::Object Proxy::Create(Napi::Env env, const Napi::CallbackInfo &info)
{
    Napi::EscapableHandleScope scope(env);
    const std::initializer_list<napi_value> arglist = {info[0]};
    Napi::Object obj = env.GetInstanceData<Napi::FunctionReference>()->New(arglist);

    return scope.Escape(napi_value(obj)).ToObject();
}

//----------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Node.js Proxy Layer.
//
// Mapping the Node-addon-api to the FTSDK APIs.
//----------------------------------------------------------------------------------------------
Napi::Object Proxy::Initialize(Napi::Env env, Napi::Object exports)
{
    Napi::Function func = DefineClass(env, "Proxy",
    {
        //- Add to register my function -//
        //
        // prototype: InstanceMethod("Function name when calling from Node.js code.", "Defined function name in C++ code.")
        InstanceMethod("ftcoreStartProcess", &Proxy::ftcoreStartProcess),
        InstanceMethod("ftcoreStartTriggerWithParam", &Proxy::ftcoreStartTriggerWithParam),
        InstanceMethod("ftcoreStartTriggerWithSetup", &Proxy::ftcoreStartTriggerWithSetup),
        InstanceMethod("ftcoreExitProcess", &Proxy::ftcoreExitProcess),
        InstanceMethod("ftcoreExitTrigger", &Proxy::ftcoreExitTrigger),
        InstanceMethod("ftcoreSendMessage", &Proxy::ftcoreSendMessage),
        InstanceMethod("ftcoreSetWindowState", &Proxy::ftcoreSetWindowState),
        InstanceMethod("ftcoreLoadDefaultCategory", &Proxy::ftcoreLoadDefaultCategory),
        InstanceMethod("ftcoreSetCustomCategory", &Proxy::ftcoreSetCustomCategory),
        InstanceMethod("ftcoreSetCustomCategoryFromFile", &Proxy::ftcoreSetCustomCategoryFromFile),
    });

    Napi::FunctionReference* fr = new Napi::FunctionReference();
    *fr = Napi::Persistent(func);
    env.SetInstanceData(fr);
    
    exports.Set("Proxy", func);
    return exports;
}

//----------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Node.js Proxy Layer.
//
// Construct FTSDKWrapper.
//----------------------------------------------------------------------------------------------
Proxy::Proxy(const Napi::CallbackInfo &info)
    : Napi::ObjectWrap<Proxy>(info)
{
    m_wrapper = new FTSDKWrapper();
};

//----------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Node.js Proxy Layer.
//
// Destruct FTSDKWrapper.
//----------------------------------------------------------------------------------------------
Proxy::~Proxy()
{
    delete m_wrapper;
    m_wrapper = nullptr;
};


//----------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Node.js Proxy Layer.
//
// Usage of ftcore_StartProcess() within FTSDKWrapper.
//----------------------------------------------------------------------------------------------
Napi::Value Proxy::ftcoreStartProcess(const Napi::CallbackInfo &info)
{
    // Convert node-addon-api to C++
    Napi::Env env = info.Env();
    std::string servername = info[0].As<Napi::String>().ToString();
    int portnumber = info[1].As<Napi::Number>().Int32Value();

    // Call FTSDKWrapper API
    int result = m_wrapper->ftcore_StartProcess(servername, portnumber);
    return Napi::Number::New(env, result);
}

//----------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Node.js Proxy Layer.
//
// Usage of ftcore_StartTriggerWithParam() within FTSDKWrapper.
//----------------------------------------------------------------------------------------------
Napi::Value Proxy::ftcoreStartTriggerWithParam(const Napi::CallbackInfo &info)
{
    // Convert node-addon-api to C++
    Napi::Env env = info.Env();
    std::string serverpath = info[0].As<Napi::String>().ToString();
    std::string servername = info[1].As<Napi::String>().ToString();
    int portnumber = info[2].As<Napi::Number>().Int32Value();
    int viewstate = info[3].As<Napi::Number>().Int32Value();
    int topmost = info[4].As<Napi::Number>().Int32Value();

    // Call FTSDKWrapper API
    int result = m_wrapper->ftcore_StartTriggerWithParam(serverpath, servername, portnumber, viewstate, topmost);
    return Napi::Number::New(env, result);
}

//----------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Node.js Proxy Layer.
//
// Usage of ftcore_StartTriggerWithSetup() within FTSDKWrapper.
//----------------------------------------------------------------------------------------------
Napi::Value Proxy::ftcoreStartTriggerWithSetup(const Napi::CallbackInfo &info)
{
    // Convert node-addon-api to C++
    Napi::Env env = info.Env();
    std::string serverpath = info[0].As<Napi::String>().ToString();
    std::string servername = info[1].As<Napi::String>().ToString();
    int portnumber = info[2].As<Napi::Number>().Int32Value();

    // Call FTSDKWrapper API
    int result = m_wrapper->ftcore_StartTriggerWithSetup(serverpath, servername, portnumber);
    return Napi::Number::New(env, result);
}

//----------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Node.js Proxy Layer.
//
// Usage of ftcore_ExitProcess() within FTSDKWrapper.
//----------------------------------------------------------------------------------------------
Napi::Value Proxy::ftcoreExitProcess(const Napi::CallbackInfo &info)
{
    // Convert node-addon-api to C++
    Napi::Env env = info.Env();

    // Call FTSDKWrapper API
    int result = m_wrapper->ftcore_ExitProcess();
    return Napi::Number::New(env, result);
}

//----------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Node.js Proxy Layer.
//
// Usage of ftcore_ExitTrigger() within FTSDKWrapper.
//----------------------------------------------------------------------------------------------
Napi::Value Proxy::ftcoreExitTrigger(const Napi::CallbackInfo &info)
{
    // Convert node-addon-api to C++
    Napi::Env env = info.Env();

    // Call FTSDKWrapper API
    int result = m_wrapper->ftcore_ExitTrigger();
    return Napi::Number::New(env, result);
}

//----------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Node.js Proxy Layer.
//
// Usage of ftcore_SendMessage() within FTSDKWrapper.
//----------------------------------------------------------------------------------------------
Napi::Value Proxy::ftcoreSendMessage(const Napi::CallbackInfo &info)
{
    // Convert node-addon-api to C++
    Napi::Env env = info.Env();
    std::string path = info[0].As<Napi::String>().ToString();
    std::string category = info[1].As<Napi::String>().ToString();
    std::string severity = info[2].As<Napi::String>().ToString();
    std::string message = info[3].As<Napi::String>().ToString();

    // Call FTSDKWrapper API
    int result = m_wrapper->ftcore_SendMessage(path, category, severity, message);
    return Napi::Number::New(env, result);
}

//----------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Node.js Proxy Layer.
//
// Usage of ftcore_SetWindowState() within FTSDKWrapper.
//----------------------------------------------------------------------------------------------
Napi::Value Proxy::ftcoreSetWindowState(const Napi::CallbackInfo &info)
{
    // Convert node-addon-api to C++
    Napi::Env env = info.Env();
    int viewstate = info[0].As<Napi::Number>().Int32Value();

    // Call FTSDKWrapper API
    int result = m_wrapper->ftcore_SetWindowState(viewstate);
    return Napi::Number::New(env, result);
}

//----------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Node.js Proxy Layer.
//
// Usage of ftcore_LoadDefaultCategory() within FTSDKWrapper.
//----------------------------------------------------------------------------------------------
Napi::Value Proxy::ftcoreLoadDefaultCategory(const Napi::CallbackInfo &info)
{
    // Convert node-addon-api to C++
    Napi::Env env = info.Env();

    // Call FTSDKWrapper API
    int result = m_wrapper->ftcore_LoadDefaultCategory();
    return Napi::Number::New(env, result);
}

//----------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Node.js Proxy Layer.
//
// Usage of ftcore_SetCustomCategory() within FTSDKWrapper.
//----------------------------------------------------------------------------------------------
Napi::Value Proxy::ftcoreSetCustomCategory(const Napi::CallbackInfo &info)
{
    // Convert node-addon-api to C++
    Napi::Env env = info.Env();
    std::string categories = info[0].As<Napi::String>().ToString();

    // Call FTSDKWrapper API
    int result = m_wrapper->ftcore_SetCustomCategory(categories);
    return Napi::Number::New(env, result);
}

//----------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Node.js Proxy Layer.
//
// Usage of ftcore_SetCustomCategoryFromFile() within FTSDKWrapper.
//----------------------------------------------------------------------------------------------
Napi::Value Proxy::ftcoreSetCustomCategoryFromFile(const Napi::CallbackInfo &info)
{
    // Convert node-addon-api to C++
    Napi::Env env = info.Env();
    std::string filename = info[0].As<Napi::String>().ToString();

    // Call FTSDKWrapper API
    int result = m_wrapper->ftcore_SetCustomCategoryFromFile(filename);
    return Napi::Number::New(env, result);
}