#include <iostream>
#include <conio.h>
#include <windows.h>
#include <dbghelp.h>
#include <psapi.h>
#include "framework.h"
#pragma comment(lib, "version.lib")
#pragma comment(lib, "dbghelp.lib")
#pragma comment(lib, "psapi.lib" )

// For GUI Window
#include "DialogWindow.h"

// For using FTSDK
#include "..\FTSDK\h\ftsdk_core.h"

#ifdef _DEBUG
#pragma comment(lib, "..\\ftsdk\\lib\\ft_cli_d.lib")
#pragma comment(lib, "..\\ftsdk\\lib\\ft_rmt_d.lib")
#else
#pragma comment(lib, "..\\ftsdk\\lib\\ft_cli.lib")
#pragma comment(lib, "..\\ftsdk\\lib\\ft_rmt.lib")
#endif


#pragma region Definitions
//----------------------------------------------------------------------------------
//
// Default category list.
//
//----------------------------------------------------------------------------------
enum DefaultCategory
{
	NON,				// No categorized
	APP,				// Application
	SYSTEM,				// System
	USER,				// User operation
	UI,					// GUI operation
	WF,					// Work flow
	DEVICE,				// Device
	DEBUG,				// Debug
	STEP,				// Step
	EVENT,				// Event
	COMM,				// Communication port

	NUMBER_OF_CATEGORY	// Number of category
};

//----------------------------------------------------------------------------------
//
// Logging severity list.
//
//----------------------------------------------------------------------------------
enum Severity
{
	NONE,				// No setting
	INFO,				// Information
	NOTICE,				// Notice
	WARNING,			// Warning
	ERR,				// Normal error
	FATAL,				// Fatal error

	NUMBER_OF_SEVERITY	// Number of severity
};
#pragma endregion Definitions

#pragma region Prototype
//----------------------------------------------------------------------------------
//
// Prototype
//
//----------------------------------------------------------------------------------
char* ConvertString(FTCORE_RESULT result, char* buffer);
char* ConvertString(DefaultCategory category, char* buffer);
char* ConvertString(Severity category, char* buffer);
void setInitialState(HWND hdlg);
void setIdleState(HWND hdlg);
void setProcessingState(HWND hdlg);
void setConnectionSettings(HWND hdlg, BOOL state);
char* GetStackFrame(long depth, char* buffer);
DWORD WINAPI runStressTest(void* args);
void progressDispatcher(HWND hdlg, int repcount);
void finishDispatcher(HWND hdlg);
void errorDispatcher(HWND hdlg, char* error);

#pragma endregion Prototype

#pragma region Dialog Event
//----------------------------------------------------------------------------------
//
// Dialog initialize.
//
// hdlg			[IN]	: Window handle of dialog.
//
//----------------------------------------------------------------------------------
void OnInitDialog(HWND hdlg)
{
	// Window size
	SetWindowPos(hdlg, NULL, 0, 0, 654, 630, (SWP_NOZORDER | SWP_NOOWNERZORDER | SWP_NOMOVE));

	// Server default settings.
	::SetDlgItemTextA(hdlg, IDC_TXT_SERVER_NAME, "localhost");
	::SetDlgItemTextA(hdlg, IDC_TXT_PORT_NUMBER, "50500");

	// Buffer for string convert.
	char buffer[MAX_PATH] = {};

	// Category combobox
	for (int index = 0; index < DefaultCategory::NUMBER_OF_CATEGORY; index++)
	{
		AddComboboxItem(hdlg, IDC_CMB_CATEGORY, ConvertString((DefaultCategory)index, buffer));
	}

	// Severity combobox
	for (int index = 0; index < Severity::NUMBER_OF_SEVERITY; index++)
	{
		AddComboboxItem(hdlg, IDC_CMB_SEVERITY, ConvertString((Severity)index, buffer));
	}

	// Repetition/Interval
	int maxvalue = 65535;
	char chrmax[MAX_PATH] = {}; ::sprintf_s(chrmax, MAX_PATH, "(%d maximum)", maxvalue);
	SetDlgItemTextA(hdlg, IDC_LBL_REPETITION_MAX, chrmax);
	SetSpinValueRange(hdlg, IDC_SPN_REPETITION, 100, 10, maxvalue);
	SetSpinValueRange(hdlg, IDC_SPN_INTERVAL, 200, 0, 1000);

	// Logging output initial message
	SetDlgItemTextA(hdlg, IDC_TXT_MESSAGE, "This is load Test for LoggingFoot.");
	OnTextChange(hdlg, IDC_TXT_MESSAGE);

	// Initial state
	setInitialState(hdlg);
}

//----------------------------------------------------------------------------------
//
// Dialog Exiting.
//
// hdlg			[IN]	: Window handle of dialog.
//
//----------------------------------------------------------------------------------
void OnExitDialog(HWND hdlg)
{
	FTCORE_ExitProcess();
}
#pragma endregion Dialog Event

#pragma region GUI Controls State
//----------------------------------------------------------------------------------
//
// GUI button initial state.
//
// hdlg			[IN]	: Window handle of dialog.
//
//----------------------------------------------------------------------------------
void setInitialState(HWND hdlg)
{
	::EnableWindow(GetDlgItem(hdlg, IDC_BTN_SEND), FALSE);
	::EnableWindow(GetDlgItem(hdlg, IDC_BTN_START), FALSE);
	::EnableWindow(GetDlgItem(hdlg, IDC_BTN_STOP), FALSE);
	::EnableWindow(GetDlgItem(hdlg, IDC_BTN_CONNECT), TRUE);
	::EnableWindow(GetDlgItem(hdlg, IDC_BTN_DISCON), FALSE);
	::SetDlgItemTextA(hdlg, IDC_LBL_STATUS, "- Stopped");
}

//----------------------------------------------------------------------------------
//
// GUI button idle state.
//
// hdlg			[IN]	: Window handle of dialog.
//
//----------------------------------------------------------------------------------
void setIdleState(HWND hdlg)
{
	setConnectionSettings(hdlg, TRUE);
	::EnableWindow(GetDlgItem(hdlg, IDC_BTN_SEND), TRUE);
	::EnableWindow(GetDlgItem(hdlg, IDC_BTN_START), TRUE);
	::EnableWindow(GetDlgItem(hdlg, IDC_BTN_STOP), FALSE);
	::EnableWindow(GetDlgItem(hdlg, IDC_BTN_CONNECT), FALSE);
	::EnableWindow(GetDlgItem(hdlg, IDC_BTN_DISCON), TRUE);
	::SetDlgItemTextA(hdlg, IDC_LBL_STATUS, "- Stopped");
}

//----------------------------------------------------------------------------------
//
// GUI button idle state.
//
// hdlg			[IN]	: Window handle of dialog.
//
//----------------------------------------------------------------------------------
void setProcessingState(HWND hdlg)
{
	::EnableWindow(GetDlgItem(hdlg, IDC_BTN_SEND), TRUE);
	::EnableWindow(GetDlgItem(hdlg, IDC_BTN_START), FALSE);
	::EnableWindow(GetDlgItem(hdlg, IDC_BTN_STOP), TRUE);
	::EnableWindow(GetDlgItem(hdlg, IDC_BTN_DISCON), FALSE);
	setConnectionSettings(hdlg, FALSE);
	::EnableWindow(GetDlgItem(hdlg, IDC_SPN_INTERVAL), TRUE);
	::EnableWindow(GetDlgItem(hdlg, IDC_SPN_REPETITION), FALSE);
	::EnableWindow(GetDlgItem(hdlg, IDC_BTN_STOP), TRUE);
	::EnableWindow(GetDlgItem(hdlg, IDC_BTN_START), FALSE);
	::SetDlgItemTextA(hdlg, IDC_LBL_STATUS, "- Running..");
}

//----------------------------------------------------------------------------------
//
// GUI button idle state.
//
// hdlg			[IN]	: Window handle of dialog.
//
//----------------------------------------------------------------------------------
void setConnectionSettings(HWND hdlg, BOOL state)
{
	::EnableWindow(GetDlgItem(hdlg, IDC_TXT_SERVER_NAME), state);
	::EnableWindow(GetDlgItem(hdlg, IDC_TXT_PORT_NUMBER), state);
	::EnableWindow(GetDlgItem(hdlg, IDC_BTN_DISCON), state);
	::EnableWindow(GetDlgItem(hdlg, IDC_BTN_CONNECT), state);
}
#pragma endregion GUI Controls State

#pragma region Server Program Connection Settings
//----------------------------------------------------------------------------------
//
// Disconnect from server.
//
// hdlg			[IN]	: Window handle of dialog.
// idc			[IN]	: Control ID of this button.(IDC_BTN_DISCON)
//
//----------------------------------------------------------------------------------
void OnButtonDisconClick(HWND hdlg, int idc)
{
	FTCORE_ExitProcess();
	setInitialState(hdlg);
}

//----------------------------------------------------------------------------------
//
// Connect to server.
//
// hdlg			[IN]	: Window handle of dialog.
// idc			[IN]	: Control ID of this button.(IDC_BTN_CONNECT)
//
//----------------------------------------------------------------------------------
void OnButtonConnectClick(HWND hdlg, int idc)
{
	char server[MAX_PATH]; ::GetDlgItemTextA(hdlg, IDC_TXT_SERVER_NAME, server, MAX_PATH);
	int port = ::GetDlgItemInt(hdlg, IDC_TXT_PORT_NUMBER, FALSE, FALSE);
	FTCORE_RESULT result = FTCORE_StartProcess(server, port);

	if (result == FTCORE_RESULT::FTCORE_SUCCESS)
	{
		// Connected button state
		setIdleState(hdlg);
	}
	else
	{
		char msg[MAX_PATH] = {}; 
		char cnv[MAX_PATH] = {};
		sprintf_s(msg, MAX_PATH,
			"Error occurred. Code=%s.\nPlease make sure LoggingFoot Server is running.", ConvertString(result, cnv));
			
		::MessageBoxA(hdlg, msg, "Sample_UnitTest", 0);
	}
}
#pragma endregion Server Program Connection Settings

#pragma region Logging Message Out
//----------------------------------------------------------------------------------
//
// Randomize category selection.
//
// hdlg			[IN]	: Window handle of dialog.
// idc			[IN]	: Control ID of this checkbox.(IDC_CHK_RND_CATEGORY)
// check		[IN]	: Random state (true: random / false: not)
//
//----------------------------------------------------------------------------------
void OnCheckRandCategoryClick(HWND hdlg, int idc, bool check)
{
	BOOL enable = check ? FALSE : TRUE;
	::EnableWindow(GetDlgItem(hdlg, IDC_CMB_CATEGORY), enable);
}

//----------------------------------------------------------------------------------
//
// Randomize severity selection.
//
// hdlg			[IN]	: Window handle of dialog.
// idc			[IN]	: Control ID of this checkbox.(IDC_CHK_RND_SEVERITY)
// check		[IN]	: Random state (true: random / false: not)
//
//----------------------------------------------------------------------------------
void OnCheckRandSeverityClick(HWND hdlg, int idc, bool check)
{
	BOOL enable = check ? FALSE : TRUE;
	::EnableWindow(GetDlgItem(hdlg, IDC_CMB_SEVERITY), enable);
}

//----------------------------------------------------------------------------------
//
// Send logging message.
//
// hdlg			[IN]	: Window handle of dialog.
// idc			[IN]	: Control ID of this button.(IDC_BTN_SEND)
//
//----------------------------------------------------------------------------------
void OnButtonSendClick(HWND hdlg, int idc)
{
	// Index of earch enum;
	int index = 0;

	// Get caller method.
	char path[MAX_PATH] = {};
	GetStackFrame(0, path);

	// Category
	if (GetCheckboxState(hdlg, IDC_CHK_RND_CATEGORY))
	{
		int categorylen = DefaultCategory::NUMBER_OF_CATEGORY;
		index = ::rand() % categorylen;
	}
	else index = GetComboboxItem(hdlg, IDC_CMB_CATEGORY);
	char category[MAX_PATH] = {}; ConvertString((DefaultCategory)index, category);

	// Severity
	if (GetCheckboxState(hdlg, IDC_CHK_RND_SEVERITY))
	{
		int severitylen = Severity::NUMBER_OF_SEVERITY;
		index = ::rand() % severitylen;
	}
	else index = GetComboboxItem(hdlg, IDC_CMB_SEVERITY);
	char severity[MAX_PATH] = {}; ConvertString((Severity)index, severity);

	// Logging message
	char message[MAX_MESSAGE_SIZE] = {}; GetDlgItemTextA(hdlg, IDC_TXT_MESSAGE, message, MAX_MESSAGE_SIZE);

	// Call LoggingFoot API
	FTCORE_SendMessage(path, category, severity, message);
}
#pragma endregion Logging Message Out

#pragma region Stress test
//----------------------------------------------------------------------------------
// Handle of runStressTest() thread.
//----------------------------------------------------------------------------------
HANDLE g_thread;

//----------------------------------------------------------------------------------
// Flag of thread running.
//----------------------------------------------------------------------------------
bool g_IsRunning;

//----------------------------------------------------------------------------------
//
// Start stress test.
//
// hdlg			[IN]	: Window handle of dialog.
// idc			[IN]	: Control ID of this button.(IDC_BTN_START)
//
//----------------------------------------------------------------------------------
void OnButtonStartClick(HWND hdlg, int idc)
{
	setProcessingState(hdlg);
	DWORD thid;
	g_thread = ::CreateThread(NULL, 0, runStressTest, (void*)hdlg, 0, &thid);
}

//----------------------------------------------------------------------------------
//
// Stop stress test.
//
// hdlg			[IN]	: Window handle of dialog.
// idc			[IN]	: Control ID of this button.(IDC_BTN_STOP)
//
//----------------------------------------------------------------------------------
void OnButtonStopClick(HWND hdlg, int idc)
{
	g_IsRunning = false;
}

//----------------------------------------------------------------------------------
//
// Stop stress test.
//
// hdlg			[IN]	: Window handle of dialog.
// idc			[IN]	: Control ID of this button.(IDC_TXT_MESSAGE)
//
//----------------------------------------------------------------------------------
void OnTextChange(HWND hdlg, int idc)
{
	char msg[MAX_MESSAGE_SIZE] = {}; GetDlgItemTextA(hdlg, idc, msg, MAX_MESSAGE_SIZE);
	int len = ::strlen(msg);
	char txt[MAX_PATH] = {};
	::sprintf_s(txt, MAX_PATH, "%d", len, MAX_PATH);

	SetDlgItemTextA(hdlg, IDC_STATIC_MSG_BYTES, txt);
}

//----------------------------------------------------------------------------------
//
// Thread of logging process.
//
// args			[IN]	: Window handle of dialog.(HWND)
//
//----------------------------------------------------------------------------------
DWORD WINAPI runStressTest(void* args)
{
	HWND hdlg = (HWND)args;

	// Index of earch enum;
	int index = 0;
	
	// Repetition count.
	BOOL btranslate = TRUE;
	int repetition = GetDlgItemInt(hdlg, IDC_TXT_REPETITION, &btranslate, TRUE);

	// Get category string.
	bool israndcat = GetCheckboxState(hdlg, IDC_CHK_RND_CATEGORY);
	char category[MAX_PATH] = {}; 
	index = GetComboboxItem(hdlg, IDC_CMB_CATEGORY);
	ConvertString((DefaultCategory)index, category);
	
	// Get severity string.
	bool israndsev = GetCheckboxState(hdlg, IDC_CHK_RND_SEVERITY);
	char severity[MAX_PATH] = {};
	index = GetComboboxItem(hdlg, IDC_CMB_SEVERITY);
	ConvertString((DefaultCategory)index, severity);

	// Send message.
	char msg[MAX_MESSAGE_SIZE] = {}; GetDlgItemTextA(hdlg, IDC_TXT_MESSAGE, msg, MAX_MESSAGE_SIZE);

	// Init progress.
	SetProgressRange(hdlg, IDC_PRG_PROGRESS, 0, repetition, 1);

	int repcount = 1;
	g_IsRunning = true;
	while (g_IsRunning && repcount <= repetition)
	{
		// Add the number of repetitions to the message.
		char message[MAX_MESSAGE_SIZE] = {};
		::snprintf(message, MAX_MESSAGE_SIZE, "Rep.%d. %s", repcount, msg);

		// Get call stack.
		char path[MAX_PATH] = {}; 
		GetStackFrame(0, path);

		// For category random.
		if (israndcat)
		{
			index = ::rand() % DefaultCategory::NUMBER_OF_CATEGORY;
			ConvertString((DefaultCategory)index, category);
		}

		// For severity random.
		if (israndsev)
		{
			index = ::rand() % Severity::NUMBER_OF_SEVERITY;
			ConvertString((Severity)index, severity);
		}

		// Call LoggingFoot API
		FTCORE_RESULT result = FTCORE_SendMessage(path, category, severity, message);
		
		// Error occurred
		if (result == FTCORE_RESULT::FTCORE_SUCCESS)
		{
			// progress update to UI
			progressDispatcher(hdlg, (repcount / (double)repetition) * 100.0);
			repcount++;

			// Interval
			int interval = GetDlgItemInt(hdlg, IDC_TXT_INTERVAL, &btranslate, TRUE);
			if(interval > 0) ::Sleep(interval);
		}
		else
		{
			char error[MAX_PATH] = {}; 
			::sprintf_s(error, MAX_PATH, "Error occurred. Result: %s", ConvertString(result, error));
			errorDispatcher(hdlg, error);
			g_IsRunning = false;
			break;
		}
	}

	// Finish
	finishDispatcher(hdlg);

	return 0;
}

//----------------------------------------------------------------------------------
//
// Update progress to UI.
//
// hdlg			[IN]	: Window handle of dialog.(HWND)
// repcount		[IN]	: current count
//
//----------------------------------------------------------------------------------
void progressDispatcher(HWND hdlg, int repcount)
{
	char progress[MAX_PATH] = {}; ::sprintf_s(progress, MAX_PATH, "%d%%", repcount);
	SetDlgItemTextA(hdlg, IDC_LBL_PROGRESS, progress);
	SetProgressValue(hdlg, IDC_PRG_PROGRESS);
}

//----------------------------------------------------------------------------------
//
// Thread of logging process finished.
//
// hdlg			[IN]	: Window handle of dialog.(HWND)
// error		[IN]	: Error message
//
//----------------------------------------------------------------------------------
void errorDispatcher(HWND hdlg, char* error)
{
	setIdleState(hdlg);
	::MessageBoxA(hdlg, error, "Sample_UnitTest", 0);
}

//----------------------------------------------------------------------------------
//
// Thread of logging process finished.
//
// hdlg			[IN]	: Window handle of dialog.(HWND)
//
//----------------------------------------------------------------------------------
void finishDispatcher(HWND hdlg)
{
	setIdleState(hdlg);
}
#pragma endregion Stress test

#pragma region Utilities
//----------------------------------------------------------------------------------
//
// Convert FTCORE_RESULT to String.
//
// result		[IN]	: Result of Called FTCORE API.
// buffer		[OUT]	: For return buffer
//
// return				: Pointer of buffer.
//----------------------------------------------------------------------------------
char* ConvertString(FTCORE_RESULT result, char* buffer)
{
	switch (result)
	{
		//-- Sucess --//

	case FTCORE_RESULT::FTCORE_SUCCESS:
		::sprintf_s(buffer, MAX_PATH, "FTCORE_SUCCESS"); break;

		//-- Events --//

	case FTCORE_RESULT::FTCORE_EVT_CLIENT_DETECTDISCON:
		::sprintf_s(buffer, MAX_PATH, "FTCORE_SUCCESS"); break;

	case FTCORE_RESULT::FTCORE_EVT_CLIENT_RECEIVED:
		::sprintf_s(buffer, MAX_PATH, "FTCORE_EVT_CLIENT_RECEIVED"); break;

		//-- Errors --//

	case FTCORE_RESULT::FTCORE_ERR_CLIENT_NOTSERVER:
		::sprintf_s(buffer, MAX_PATH, "FTCORE_ERR_CLIENT_NOTSERVER"); break;

	case FTCORE_RESULT::FTCORE_ERR_CLIENT_FAILSERVER:
		::sprintf_s(buffer, MAX_PATH, "FTCORE_ERR_CLIENT_FAILSERVER"); break;

	case FTCORE_RESULT::FTCORE_ERR_CLIENT_NOEXIST:
		::sprintf_s(buffer, MAX_PATH, "FTCORE_ERR_CLIENT_NOEXIST"); break;

	case FTCORE_RESULT::FTCORE_ERR_CLIENT_ALREADY:
		::sprintf_s(buffer, MAX_PATH, "FTCORE_ERR_CLIENT_ALREADY"); break;

	case FTCORE_RESULT::FTCORE_ERR_CLIENT_PARAMETER:
		::sprintf_s(buffer, MAX_PATH, "FTCORE_ERR_CLIENT_PARAMETER"); break;

	case FTCORE_RESULT::FTCORE_ERR_CLIENT_HOSTINFO:
		::sprintf_s(buffer, MAX_PATH, "FTCORE_ERR_CLIENT_HOSTINFO"); break;

	case FTCORE_RESULT::FTCORE_ERR_CLIENT_SOCKET:
		::sprintf_s(buffer, MAX_PATH, "FTCORE_ERR_CLIENT_SOCKET"); break;

	case FTCORE_RESULT::FTCORE_ERR_CLIENT_REFUSED:
		::sprintf_s(buffer, MAX_PATH, "FTCORE_ERR_CLIENT_REFUSED"); break;

	case FTCORE_RESULT::FTCORE_ERR_CLIENT_UNREACHED:
		::sprintf_s(buffer, MAX_PATH, "FTCORE_ERR_CLIENT_UNREACHED"); break;

	case FTCORE_RESULT::FTCORE_ERR_CLIENT_CONNECT:
		::sprintf_s(buffer, MAX_PATH, "FTCORE_ERR_CLIENT_CONNECT"); break;

	case FTCORE_RESULT::FTCORE_ERR_CLIENT_ESTABLISH:
		::sprintf_s(buffer, MAX_PATH, "FTCORE_ERR_CLIENT_ESTABLISH"); break;

	case FTCORE_RESULT::FTCORE_ERR_CLIENT_MESSAGESEND:
		::sprintf_s(buffer, MAX_PATH, "FTCORE_ERR_CLIENT_MESSAGESEND"); break;

	case FTCORE_RESULT::FTCORE_ERR_CLIENT_MESSAGERECV:
		::sprintf_s(buffer, MAX_PATH, "FTCORE_ERR_CLIENT_MESSAGERECV"); break;

	case FTCORE_RESULT::FTCORE_ERR_CLIENT_CONNUNKNOWN:
		::sprintf_s(buffer, MAX_PATH, "FTCORE_ERR_CLIENT_CONNUNKNOWN"); break;

	default:
		::sprintf_s(buffer, MAX_PATH, "FTCORE_UNKNOWN_STATE");
		break;
	}

	return buffer;
}

//----------------------------------------------------------------------------------
//
// Convert Enum of Category to String.
//
// result		[IN]	: Enum of Category
// buffer		[OUT]	: For return buffer
//
// return				: Pointer of buffer.
//----------------------------------------------------------------------------------
char* ConvertString(DefaultCategory category, char* buffer)
{
	switch (category)
	{
	case DefaultCategory::NON:
		::sprintf_s(buffer, MAX_PATH, "NON"); break;

	case DefaultCategory::APP:
		::sprintf_s(buffer, MAX_PATH, "APP"); break;

	case DefaultCategory::SYSTEM:
		::sprintf_s(buffer, MAX_PATH, "SYSTEM"); break;

	case DefaultCategory::USER:
		::sprintf_s(buffer, MAX_PATH, "USER"); break;

	case DefaultCategory::UI:
		::sprintf_s(buffer, MAX_PATH, "UI"); break;

	case DefaultCategory::WF:
		::sprintf_s(buffer, MAX_PATH, "WF"); break;

	case DefaultCategory::DEVICE:
		::sprintf_s(buffer, MAX_PATH, "DEVICE"); break;

	case DefaultCategory::DEBUG:
		::sprintf_s(buffer, MAX_PATH, "DEBUG"); break;

	case DefaultCategory::STEP:
		::sprintf_s(buffer, MAX_PATH, "STEP"); break;

	case DefaultCategory::EVENT:
		::sprintf_s(buffer, MAX_PATH, "EVENT"); break;

	case DefaultCategory::COMM:
		::sprintf_s(buffer, MAX_PATH, "COMM"); break;
	}

	return buffer;
}

//----------------------------------------------------------------------------------
//
// Convert Enum of Severity to String.
//
// result		[IN]	: Enum of Severity
// buffer		[OUT]	: For return buffer
//
// return				: Pointer of buffer.
//----------------------------------------------------------------------------------
char* ConvertString(Severity severity, char* buffer)
{
	switch (severity)
	{
	case Severity::NONE:
		::sprintf_s(buffer, MAX_PATH, "NON"); break;

	case Severity::INFO:
		::sprintf_s(buffer, MAX_PATH, "INFO"); break;

	case Severity::NOTICE:
		::sprintf_s(buffer, MAX_PATH, "NOTICE"); break;

	case Severity::WARNING:
		::sprintf_s(buffer, MAX_PATH, "WARNING"); break;

	case Severity::ERR:
		::sprintf_s(buffer, MAX_PATH, "ERR"); break;

	case Severity::FATAL:
		::sprintf_s(buffer, MAX_PATH, "FATAL"); break;
	}

	return buffer;
}

//----------------------------------------------------------------------------------
//
// Get stack frame information.
// It can be obtained correctly when in debug mode, but not in release mode.
//
// depth		[IN]	: stack frame depth
// buffer		[OUT]	: For return buffer
//
// return				: Pointer of buffer.
//----------------------------------------------------------------------------------
char* GetStackFrame(long depth, char* buffer)
{
	CONTEXT context;
	::RtlCaptureContext(&context);

	// include self (+1) and parent (+1) = (+2)
	int stacknum = depth + 2;

	STACKFRAME sf; ::ZeroMemory(&sf, sizeof(sf));
	sf.AddrPC.Offset = context.Rip;
	sf.AddrStack.Offset = context.Rsp;
	sf.AddrFrame.Offset = context.Rsp;
	sf.AddrPC.Mode = AddrModeFlat;
	sf.AddrStack.Mode = AddrModeFlat;
	sf.AddrFrame.Mode = AddrModeFlat;

	HANDLE pch = GetCurrentProcess();
	HANDLE pth = GetCurrentThread();

	BOOL bresult = TRUE;
	int count = 0;

	//load symbols
	::SymInitialize(pch, NULL, TRUE);

	while (bresult && count < stacknum)
	{
		bresult = ::StackWalk64(IMAGE_FILE_MACHINE_AMD64, pch, pth, &sf, (void*)&context, NULL, SymFunctionTableAccess, SymGetModuleBase, NULL);

		if (bresult)
		{
			const int MaxNameLen = 256;
			char symbuff[sizeof(SYMBOL_INFO) + MAX_SYM_NAME * sizeof(TCHAR)];
			char name[MaxNameLen];
			char module[MaxNameLen];
			PSYMBOL_INFO psymbol = (PSYMBOL_INFO)symbuff;
			psymbol->SizeOfStruct = sizeof(SYMBOL_INFO);
			psymbol->MaxNameLen = MAX_SYM_NAME;
			DWORD64 displacement;
			::SymFromAddr(pch, (ULONG64)sf.AddrPC.Offset, &displacement, psymbol);

			//char linebuff[sizeof(IMAGEHLP_LINE64)];
			//DWORD disp;
			//PIMAGEHLP_LINE64 pimgline = (PIMAGEHLP_LINE64)linebuff;
			//pimgline->SizeOfStruct = sizeof(IMAGEHLP_LINE64);
			//bresult = ::SymGetLineFromAddr64(pch, sf.AddrPC.Offset, &disp, pimgline);

			::sprintf_s(buffer, MAX_PATH, "%s()", psymbol->Name);
		}

		count++;
	}

	return buffer;
}
#pragma endregion Utilities