//-----------------------------------------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Core published APIs
//
// Copyright © 2025 Quantyworks Software
//-----------------------------------------------------------------------------------------------------------------------------
#pragma once
#ifndef __LOGGINGFOOTCORE_H_INCLUDED__
#define __LOGGINGFOOTCORE_H_INCLUDED__

//-------------------------------------------------------------
// Definitions
//-------------------------------------------------------------
#define MAX_MESSAGE_SIZE 4097

//-------------------------------------------------------------
// LoggingFoot FTSDK API Results.
//-------------------------------------------------------------
typedef enum FTCORE_RESULT
{
    FTCORE_UNKNOWN_STATE = -1,          // The state is undefined.

    FTCORE_SUCCESS = 0,                 // Operation completed successfully.


    //-- Events --//

    FTCORE_EVT_CLIENT_DETECTDISCON = 4, // Client detected a disconnection from the server.
    FTCORE_EVT_CLIENT_RECEIVED,         // Client received data from the server.


    //-- Errors --//

    FTCORE_ERR_CLIENT_NOTSERVER = 21,   // The server is not running or does not exist.
    FTCORE_ERR_CLIENT_FAILSERVER,       // Failed to start the logging server.

    FTCORE_ERR_CLIENT_NOEXIST,          // Called before the process was created.
    FTCORE_ERR_CLIENT_ALREADY,          // The process has already been started.
    FTCORE_ERR_CLIENT_PARAMETER,        // Invalid parameters were provided.
    FTCORE_ERR_CLIENT_HOSTINFO,         // Failed to retrieve host information.
    FTCORE_ERR_CLIENT_SOCKET,           // Failed to create a socket.
    FTCORE_ERR_CLIENT_REFUSED,          // The target process is not accepting connections or refused the connection.
    FTCORE_ERR_CLIENT_UNREACHED,        // The network path to the destination is unknown.
    FTCORE_ERR_CLIENT_CONNECT,          // Connection failed due to an unknown cause.
    FTCORE_ERR_CLIENT_ESTABLISH,        // An error occurred during the connection establishment process.
    FTCORE_ERR_CLIENT_MESSAGESEND,      // Failed to send the message.
    FTCORE_ERR_CLIENT_MESSAGERECV,      // Failed to receive the message.
    FTCORE_ERR_CLIENT_CONNUNKNOWN,      // Received a connection notification from an unknown server.
} FTCORE_RESULT;

//-------------------------------------------------------------
// Received data structure.
//-------------------------------------------------------------
typedef struct FTCORE_RECEIVED_DATA
{
	// Reason for receipt notification.
	FTCORE_RESULT Reason;

	// Network host name.
	char* HostName;

	// Network host process id.
	unsigned long ProcessId;

	// Network host process name.
	char* ProcessName;

	// Network host connection id.
	char* ConnectionId;

	// Received message.
	char* Message;

} FTCORE_RECEIVED_DATA;

//-----------------------------------------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Core Export API
//
// Starts the LoggingFoot client process.
//
//
// servername		(IN)		: The server name or IP address of the destination to connect to.
// portnumber		(IN)		: The port number of the destination server (49152–65535).
//
// return			(SUCCESS)	: FTCORE_SUCCESS
//					(ONERROR)	: FTCORE_ERR_CLIENT_ALREADY
//								: FTCORE_ERR_CLIENT_SOCKET
//								: FTCORE_ERR_CLIENT_REFUSED
//								: FTCORE_ERR_CLIENT_UNREACHED
//								: FTCORE_ERR_CLIENT_CONNECT
//								: FTCORE_ERR_CLIENT_ESTABLISH
//-----------------------------------------------------------------------------------------------------------------------------
FTCORE_RESULT __stdcall FTCORE_StartProcess(char* servername, unsigned short portnumber);

//-----------------------------------------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Core Export API
//
// Terminates the client process.
// This function must be called to terminate the process if it was started using FTCORE_StartProcess().
//
//
// return			(SUCCESS)	: FTCORE_SUCCESS
//					(ONERROR)	: --------
//-----------------------------------------------------------------------------------------------------------------------------
FTCORE_RESULT __stdcall FTCORE_ExitProcess();

//-----------------------------------------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Core Export API
//
// After the server process is started with the parameters passed, the client process is started.
//
//
// serverpath		(IN)		: The full path to the destination server executable.
// servername		(IN)		: The server name or IP address to connect to.
// portnumber		(IN)		: The port number of the destination server (49152–65535).
// viewstate		(IN)		: The window state of the server process.
//								: (0: Default window size)
//								: (1: Maximized window)
//								: (2: Minimized to thw taskbar)
//								: (3: Hidden in the system tray)
//								: (4: Fully hidden)
// topmost			(IN)		: Specifies whether the window should stay on top of other windows.
//
// return			(SUCCESS)	: FTCORE_SUCCESS
//					(ONERROR)	: FTCORE_ERR_CLIENT_NOTSERVER
//								: FTCORE_ERR_CLIENT_FAILSERVER
//								: FTCORE_ERR_CLIENT_ALREADY
//								: FTCORE_ERR_CLIENT_SOCKET
//								: FTCORE_ERR_CLIENT_REFUSED
//								: FTCORE_ERR_CLIENT_UNREACHED
//								: FTCORE_ERR_CLIENT_CONNECT
//								: FTCORE_ERR_CLIENT_ESTABLISH
//-----------------------------------------------------------------------------------------------------------------------------
FTCORE_RESULT __stdcall FTCORE_StartTriggerWithParam(char* serverpath, char* servername, unsigned short portnumber, int viewstate, int topmost);

//-----------------------------------------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Core Export API
//
// Starts the client process after the server process has been launched, using the setup parameters.
//
//
// serverpath		(IN)		: The full path to the destination server executable.
// servername		(IN)		: The server name or IP address to connect to.
// portnumber		(IN)		: The port number of the destination server (49152–65535).
//
// return			(SUCCESS)	: FTCORE_SUCCESS
//					(ONERROR)	: FTCORE_ERR_CLIENT_NOTSERVER
//								: FTCORE_ERR_CLIENT_FAILSERVER
//								: FTCORE_ERR_CLIENT_ALREADY
//								: FTCORE_ERR_CLIENT_SOCKET
//								: FTCORE_ERR_CLIENT_REFUSED
//								: FTCORE_ERR_CLIENT_UNREACHED
//								: FTCORE_ERR_CLIENT_CONNECT
//								: FTCORE_ERR_CLIENT_ESTABLISH
//-----------------------------------------------------------------------------------------------------------------------------
FTCORE_RESULT __stdcall FTCORE_StartTriggerWithSetup(char* serverpath, char* servername, unsigned short portnumber);

//-----------------------------------------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Core Export API
//
// Terminates the client process after the server process has exited.
// This function must be called to terminate the process if it was started using either FTCORE_StartTriggerWithParam() or FTCORE_StartTriggerWithSetup().
// If no server process is running, only the client process will be terminated.
//
//
// return			(SUCCESS)	: FTCORE_SUCCESS
//					(ONERROR)	: --------
//-----------------------------------------------------------------------------------------------------------------------------
FTCORE_RESULT __stdcall FTCORE_ExitTrigger();

//-----------------------------------------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Core Export API
//
// Send a message to server process.
//
//
// path				(IN)		: The caller's path or source identifier.
// category			(IN)		: The log category.
// severity			(IN)		: The log severity level.
// message			(IN)		: The message content to send.
//
// return			(SUCCESS)	: FTCORE_SUCCESS
//					(ONERROR)	: FTCORE_ERR_CLIENT_NOEXIST
//								: FTCORE_ERR_CLIENT_PARAMETER
//								: FTCORE_ERR_CLIENT_MESSAGESEND
//-----------------------------------------------------------------------------------------------------------------------------
FTCORE_RESULT __stdcall FTCORE_SendMessage(char* path, char* category, char* severity, char* message);

//-----------------------------------------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Core Export API
//
// Instructs server process to change its window display state.
//
//
// state			(IN)		: 0 SHOW_WINDOW_NORMALY		Default window size
//								: 1 SHOW_WINDOW_MAXIMIZE	Maximized window
//								: 2 MINIMIZE_IN_TASKBAR		Minimized in taskbar
//								: 3 HIDE_IN_TASKTRAY		Window hidden
//
// return			(SUCCESS)	: FTCORE_SUCCESS
//					(ONERROR)	: FTCORE_ERR_CLIENT_NOEXIST
//								: FTCORE_ERR_CLIENT_PARAMETER
//								: FTCORE_ERR_CLIENT_MESSAGESEND
//-----------------------------------------------------------------------------------------------------------------------------
FTCORE_RESULT __stdcall FTCORE_SetWindowState(int state);

//-----------------------------------------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Core Export API
//
// Instructs the server process to load the default logging category values.
//
//
// return			(SUCCESS)	: FTCORE_SUCCESS
//					(ONERROR)	: FTCORE_ERR_CLIENT_NOEXIST
//								: FTCORE_ERR_CLIENT_PARAMETER
//								: FTCORE_ERR_CLIENT_MESSAGESEND
//-----------------------------------------------------------------------------------------------------------------------------
FTCORE_RESULT __stdcall FTCORE_LoadDefaultCategory();

//-----------------------------------------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Core Export API
//
// Instructs the server process to apply custom logging categories.
//
//
// categories		(IN)		: One or more category names, separated by commas if multiple.
//
// return			(SUCCESS)	: FTCORE_SUCCESS
//					(ONERROR)	: FTCORE_ERR_CLIENT_NOEXIST
//								: FTCORE_ERR_CLIENT_PARAMETER
//								: FTCORE_ERR_CLIENT_MESSAGESEND
//-----------------------------------------------------------------------------------------------------------------------------
FTCORE_RESULT __stdcall FTCORE_SetCustomCategory(char* categories);

//-----------------------------------------------------------------------------------------------------------------------------
// LoggingFoot FTSDK Core Export API
//
// Instructs the server process to apply the loaded category settings from the specified file.
//
//
// filename 		(IN)		: Full path of category setting file name.
//
// return			(SUCCESS)	: FTCORE_SUCCESS
//					(ONERROR)	: FTCORE_ERR_CLIENT_NOEXIST
//								: FTCORE_ERR_CLIENT_PARAMETER
//								: FTCORE_ERR_CLIENT_MESSAGESEND
//-----------------------------------------------------------------------------------------------------------------------------
FTCORE_RESULT __stdcall FTCORE_SetCustomCategoryFromFile(char* filename);


#endif	//  __LOGGINGFOOTCORE_H_INCLUDED__
