import win32api     # need to install pywin32
import os
import sys
import msvcrt
import random
import inspect
from pathlib import Path
from enum import Enum

import FtsdkPYWrapper as ftsdk

class DefaultCategory(Enum):
    """Default category list."""

    NON     = 0     # No categorized
    APP     = 1     # Application
    SYSTEM  = 2     # System
    USER    = 3     # User operation
    UI      = 4     # GUI operation
    WF      = 5     # Work flow
    DEVICE  = 6     # Device
    DEBUG   = 7     # Debug
    STEP    = 8     # Step
    EVENT   = 9     # Event
    COMM    = 10    # Communication port

class CustomCategory(Enum):
    """Custom category list."""

    NON         = 0 # No categorized
    C_APP       = 1 # Application
    C_SYSTEM    = 2 # System
    C_USER      = 3 # User operation
    C_UI        = 4 # GUI operation
    C_WF        = 5 # Work flow
    C_DEVICE    = 6 # Device
    C_DEBUG     = 7 # Debug
    C_STEPLOG   = 8 # Step
    C_COMM      = 9 # Communication port

class FileCustomCategory(Enum):
    """File Custom category list."""

    NON         = 0 # No categorized
    F_APP       = 1 # Application
    F_SYSTEM    = 2 # System
    F_USER      = 3 # User operation
    F_UI        = 4 # GUI operation
    F_WF        = 5 # Work flow
    F_DEVICE    = 6 # Device
    F_DEBUG     = 7 # Debug
    F_STEPLOG   = 8 # Step
    F_COMM      = 9 # Communication port

class Severity(Enum):
    """Logging severity list."""

    NON     = 0     # No setting
    INFO    = 1     # Information
    NOTICE  = 2     # Notice
    WARNING = 3     # Warning
    ERROR   = 4     # Normal error
    FATAL   = 5     # Fatal error

def getSDKVersion() -> str:
    """ Get Logging Foot SDK version. """

    version = "-----"
    try:
        # Get the path of the DLL
        pathobj = Path(__file__).parent.parent.resolve()
        dllpath = os.path.join(pathobj, "ftsdk", "bin", "ft_cli.dll")
        
        # Get the version information
        verinfo = win32api.GetFileVersionInfo(dllpath, "\\")
        ms = verinfo['FileVersionMS']
        ls = verinfo['FileVersionLS']
        version = f"{ms >> 16}.{ms & 0xFFFF}.{ls >> 16}.{ls & 0xFFFF}"

    except Exception as e:
        print(f"Error getting SDK version: {e}")
    return version

def OutputLog(message: str, categorytype: int):
    """
    Outputs a log message.

    Args:
        message      : The content of the log message to be sent.
        categorytype : The type of category to use.
                        0: DefaultCategory
                        1: CustomCategory
                        2: FileCustomCategory
    """

    # Random number generator
    rand = random.Random()

    # Get caller method
    stack = inspect.stack()
    if len(stack) > 1:
        path = stack[1].function
    else:
        path = "<unknown>"

    # Select category type
    if categorytype == 0:
        categories = [e.name for e in DefaultCategory]
    elif categorytype == 1:
        categories = [e.name for e in CustomCategory]
    else:
        categories = [e.name for e in FileCustomCategory]

    # Get log categories at random
    category = rand.choice(categories)

    # Get log severity at random
    severities = [e.name for e in Severity]
    severity = rand.choice(severities)

    # API execution
    print("> Called FTCORE_SendMessage(...)")
    result = ftsdk.FtsdkPYWrapper.FTCORE_SendMessage(path, category, severity, message)
    print(f"> result: {result.name}")
    print("")


def main():
    """ LoggingFoot sample program. """
    """ How to use FTCORE_StartProcess and FTCORE_SetDefaultCategory. """

    copyright = "Copyright (C) 2025 Quantyworks Software"
    version = getSDKVersion()

    print("**********************************************************************")
    print("")
    print("  LoggingFoot FTSDK sample program.")
    print("  How to use FTCORE_StartProcess and FTCORE_SetCustomCategory.")
    print("")
    print(f"  {copyright}")
    print(f"                                                  FTSDK ver.{version}")
    print("**********************************************************************")
    print("")
    print("First, Please start the Logging server.")
    print("And next, press any key. (Press Esc key to exit this program.)")
    print("Logging client will start.")
    print("")
    
    # Destination server information.
    server = "localhost"
    port = 50500

    # API execution result.
    result = ftsdk.FtsdkPYWrapper.FTCORE_RESULT.FTCORE_UNKNOWN_STATE

    # Select category type
    # 0: default category
    # 1: custom category
    # 2: custom category read from file
    categorytype = 0;

    # Wait for key input.
    print("> Wait for any key to be pressed ...")
    print("> ", end="", flush=True)
    key = msvcrt.getch().decode()
    print(f"{key}")
    print(f"> {key} key was pressed.")
    if key == '\x1b':   # ESC key

        #- Exit program -#
        pass

    else:

        #- Startup Logging Client -//
        print("> Called FTCORE_StartProcess()..");
        result = ftsdk.FtsdkPYWrapper.FTCORE_StartProcess(server, port)
        print(f"> result: {result.name}")
        print("")

    if result == ftsdk.FtsdkPYWrapper.FTCORE_RESULT.FTCORE_SUCCESS:

        #- Select custom category setting API -#

        print("> Press following number key, select API.");
        print("> 0 : FTCORE_LoadDefaultCategory()");
        print("> 1 : FTCORE_SetCustomCategory()");
        print("> 2 : FTCORE_SetCustomCategoryFromFile()");
        print("")

        key = '0'

        while key != '\x1b':   # ESC key

            #- Log out input keys -#
            
            print("> ", end="", flush=True)
            key = msvcrt.getch().decode()
            print(f"{key} key was pressed.")
            logmsg = f"{key} key was pressed."
            OutputLog(logmsg, categorytype)

            #- Call FTCORE_SetCustomCategory API -#

            if key == '0':
                
                # Directs loading of default category that stored in the Logging Server.
                categorytype = 0
                print("> Called FTCORE_LoadDefaultCategory()..")
                result = ftsdk.FtsdkPYWrapper.FTCORE_LoadDefaultCategory()
                print(f"> result: {result.name}")
                print("")

            elif key == '1':
                
                # Set the category defined in this program to the Logging Server.
                categorytype = 1
                
                # Concatenate category names separated by commas (with or without a comma at the end of the string).
                categories = ",".join([e.name for e in CustomCategory])

                # Set to the Logging Server process.
                print("> Called FTCORE_SetCustomCategory()..")
                result = ftsdk.FtsdkPYWrapper.FTCORE_SetCustomCategory(categories)
                print(f"> result: {result.name}")
                print("")

            elif key == '2':

                # Read the categories stored in the file and set them to the Logging Server.
                categorytype = 2
                pathobj = Path(__file__)
                file = os.path.join(pathobj.parent,"CustomCategory_Sample.catdef")
                
                # Set to the Logging Server process.
                print("> Called FTCORE_SetCustomCategoryFromFile()..")
                result = ftsdk.FtsdkPYWrapper.FTCORE_SetCustomCategoryFromFile(file)
                print(f"> result: {result.name}")
                print("")

            elif key == '\x1b':   # ESC key
                
                # Logging Client exit process.
                print("> Called FTCORE_ExitProcess()..")
                result = ftsdk.FtsdkPYWrapper.FTCORE_ExitProcess()
                print(f"> result: {result.name}")
                print("")

    # Press any key to exit program.
    print("Press any key again to exit program")
    msvcrt.getch()

if __name__ == "__main__":
    main()