﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using FTSDK_Wrapper;

namespace Sample_StressTest
{
    public partial class frmMain : Form
    {
        #region Definitions
        /// <summary>
        /// Maximum message size (in bytes).
        /// </summary>
        const int MAX_MESSAGE_SIZE = 4096;
        
        /// <summary>
        /// Default category list.
        /// </summary>
        enum DefaultCategory
        {
            /// <summary>
            /// No categorized
            /// </summary>
            NON,
            /// <summary>
            /// Application
            /// </summary>
            APP,
            /// <summary>
            /// System
            /// </summary>
            SYSTEM,
            /// <summary>
            /// User operation
            /// </summary>
            USER,
            /// <summary>
            /// GUI operation
            /// </summary>
            UI,
            /// <summary>
            /// Work flow
            /// </summary>
            WF,
            /// <summary>
            /// Device
            /// </summary>
            DEVICE,
            /// <summary>
            /// Debug
            /// </summary>
            DEBUG,
            /// <summary>
            /// Step
            /// </summary>
            STEP,
            /// <summary>
            /// Event
            /// </summary>
            EVENT,
            /// <summary>
            /// Communication port
            /// </summary>
            COMM,
        }

        /// <summary>
        /// Logging severity list.
        /// </summary>
        enum Severity
        {
            /// <summary>
            /// No setting
            /// </summary>
            NON,
            /// <summary>
            /// Information
            /// </summary>
            INFO,
            /// <summary>
            /// Notice
            /// </summary>
            NOTICE,
            /// <summary>
            /// Warning
            /// </summary>
            WARNING,
            /// <summary>
            /// Normal error
            /// </summary>
            ERROR,
            /// <summary>
            /// Fatal error
            /// </summary>
            FATAL,
        }
        #endregion Definitions

        #region Constructor
        /// <summary>
        /// Constructor
        /// </summary>
        public frmMain()
        {
            InitializeComponent();

            // Connection Settings
            this.btnConnect.Click += this.btnConnect_Click;
            this.btnDiscon.Click += this.btnDiscon_Click;

            // Category combobox
            var categories = System.Enum.GetNames(typeof(DefaultCategory));
            System.Array.ForEach(categories, item => this.cmbCategory.Items.Add(item));
            this.cmbCategory.SelectedIndex = 0;

            // Severity combobox
            var severities = System.Enum.GetNames(typeof(Severity));
            System.Array.ForEach(severities, item => this.cmbSeverity.Items.Add(item));
            this.cmbSeverity.SelectedIndex = 0;

            // Random checkbox
            this.chkCategoryRandom.Click += this.chkCategoryRandom_Click;
            this.chkSeverityRandom.Click += this.chkSeverityRandom_Click;

            // Send button
            this.btnSend.Click += this.btnSend_Click;

            // Repetition
            var repmax = 65535;
            this.lblRepetitionMax.Text = $"({repmax} maximum)";
            this.numRepetition.Minimum = 100;
            this.numRepetition.Maximum = repmax;

            // Start/Stop button
            this.btnStart.Click += this.btnStart_Click;
            this.btnStop.Click += this.btnStop_Click;
            this.btnStop.Enabled = false;

            // Logging output initial message
            this.txtMessage.Text = "This is load Test for LoggingFoot.";

            // Windows Forms Event
            this.Load += this.frmMain_Load;
            this.FormClosed += this.frmMain_Closed;
        }
        #endregion Constructor

        #region Windows Forms Event
        /// <summary>
        /// Form load
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void frmMain_Load(object sender, EventArgs e)
        {
            // Initial state
            this.setInitialState();
        }

        /// <summary>
        /// Disconnect from server and exit logging.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void frmMain_Closed(object sender, FormClosedEventArgs e)
        {
            FtsdkCSWrapper.FTCORE_ExitProcess();
        }

        /// <summary>
        /// Message text changed event
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void txtMessage_TextChanged(object sender, EventArgs e)
        {
            var ctrl = sender as System.Windows.Forms.TextBox;
            if (ctrl == null) return;

            var text = ctrl.Text;
            var bytenum = Encoding.UTF8.GetByteCount(ctrl.Text);
            if(bytenum <= MAX_MESSAGE_SIZE) this.lblMessageBytes.Text = bytenum.ToString();
            else
            {
                int byteCount = 0;
                int charCount = 0;

                foreach (char c in text)
                {
                    int charByteSize = Encoding.UTF8.GetByteCount(new[] { c });
                    if (byteCount + charByteSize > MAX_MESSAGE_SIZE)
                        break;

                    byteCount += charByteSize;
                    charCount++;
                }

                ctrl.Text = text.Substring(0, charCount);
            }
        }
        #endregion Windows Forms Event

        #region Windows Forms State
        /// <summary>
        /// GUI button initial state.
        /// </summary>
        private void setInitialState()
        {
            this.btnSend.Enabled = false;
            this.btnStart.Enabled = false;
            this.btnStop.Enabled = false;
            this.btnConnect.Enabled = true;
            this.btnDiscon.Enabled = false;
        }

        /// <summary>
        /// GUI button idle state.
        /// </summary>
        private void setIdleState()
        {
            this.btnSend.Enabled = true;
            this.btnStart.Enabled = true;
            this.btnStop.Enabled = false;
            this.btnConnect.Enabled = false;
            this.btnDiscon.Enabled = true;
            this.grpConnectionSettings.Enabled = true;
            this.pnlStressTestSettings.Enabled = true;
            this.lblStatus.Text = "- Stopped";
        }

        /// <summary>
        /// GUI button processing state.
        /// </summary>
        private void setProcessingState()
        {
            this.btnSend.Enabled = true;
            this.btnStart.Enabled = false;
            this.btnStop.Enabled = true;
            this.btnDiscon.Enabled = false;
            this.grpConnectionSettings.Enabled = false;
            this.pnlStressTestSettings.Enabled = false;
            this.lblStatus.Text = "- Running..";
        }
        #endregion Windows Forms State

        #region Server Program Connection Setttings
        /// <summary>
        /// Server Name
        /// </summary>
        private string ServerName
        {
            get { return this.txtServerName.Text; }
        }
        /// <summary>
        /// Port Number
        /// </summary>
        private int PortNumber
        {
            get
            {
                var defnum = 50500;
                var number = 0;
                if (!int.TryParse(this.txtPortNumber.Text, out number))
                {
                    number = defnum;
                }

                return number;
            }
        }

        /// <summary>
        /// Connect to server
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnConnect_Click(object sender, EventArgs e)
        {
            var server = this.ServerName;
            var port = this.PortNumber;
            var result = FtsdkCSWrapper.FTCORE_StartProcess(server, port);

            if (result == FtsdkCSWrapper.FTCORE_RESULT.FTCORE_SUCCESS)
            {
                // Connected button state
                this.setIdleState();
            }
            else
            {
                var msg = System.String.Format("Error occurred. Code={0}.\nPlease make sure LoggingFoot Server is running.", result);
                System.Windows.Forms.MessageBox.Show(msg, "Error", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            }
        }
        /// <summary>
        /// Disconnect from server
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnDiscon_Click(object sender, EventArgs e)
        {
            FtsdkCSWrapper.FTCORE_ExitProcess();

            this.setInitialState();
        }
        #endregion Server Program Connection Setttings

        #region Logging Message Output
        /// <summary>
        /// Category random checked state
        /// </summary>
        private bool IsCategoryRandom
        {
            get { return this.chkCategoryRandom.Checked; }
        }
        /// <summary>
        /// Selected Category
        /// </summary>
        private string SelectedCategory
        {
            get
            {
                if (this.IsCategoryRandom)
                {
                    var rand = new System.Random();
                    var names = System.Enum.GetNames(typeof(DefaultCategory));
                    return names[rand.Next() % names.Length];
                }
                else return this.cmbCategory.SelectedItem.ToString();
            }
        }
        /// <summary>
        /// Category Random checkbox
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void chkCategoryRandom_Click(object sender, EventArgs e)
        {
            var checkbox = sender as System.Windows.Forms.CheckBox;
            this.cmbCategory.Enabled = !checkbox.Checked;
        }

        /// <summary>
        /// Severity random checked state
        /// </summary>
        private bool IsSeverityRandom
        {
            get { return this.chkSeverityRandom.Checked; }
        }
        /// <summary>
        /// Selected Severity
        /// </summary>
        private string SelectedSeverity
        {
            get
            {
                if (this.IsSeverityRandom)
                {
                    var rand = new System.Random();
                    var names = System.Enum.GetNames(typeof(Severity));
                    return names[rand.Next() % names.Length];
                }
                else return this.cmbSeverity.SelectedItem.ToString();
            }
        }
        /// <summary>
        /// Severity Random checkbox
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void chkSeverityRandom_Click(object sender, EventArgs e)
        {
            var checkbox = sender as System.Windows.Forms.CheckBox;
            this.cmbSeverity.Enabled = !checkbox.Checked;
        }

        /// <summary>
        /// Logging output message
        /// </summary>
        private string Message
        {
            get { return this.txtMessage.Text; }
        }
        /// <summary>
        /// Send logging message.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnSend_Click(object sender, EventArgs e)
        {
            // Get call stack.
            var stack = new System.Diagnostics.StackFrame(1);
            var path = stack.GetMethod().ToString();

            // Category
            var category = this.SelectedCategory;

            // Severity
            var severity = this.SelectedSeverity;

            // Logging message
            var message = this.Message;

            // Call LoggingFoot API
            var result = FtsdkCSWrapper.FTCORE_SendMessage(path, category, severity, message);
        }
        #endregion Logging Message Output

        #region Stress test
        /// <summary>
        /// Repetition Count
        /// </summary>
        private int RepetitionCount
        {
            get { return (int)this.numRepetition.Value; }
        }
        /// <summary>
        /// Repetition Interval
        /// </summary>
        private int RepetitionInterval
        {
            get { return (int)this.numInterval.Value; }
        }

        /// <summary>
        /// Flag of thread running
        /// </summary>
        private bool IsRunning { get; set; }

        /// <summary>
        /// Start stress test
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnStart_Click(object sender, EventArgs e)
        {
            var thread = new System.Threading.Thread(new System.Threading.ParameterizedThreadStart(this.runStressTest));

            var parameters = new object[]
            {
                this.RepetitionCount,
                this.IsCategoryRandom,
                this.SelectedCategory,
                this.IsSeverityRandom,
                this.SelectedSeverity,
                this.Message,
            };

            this.setProcessingState();
            thread.Start(parameters);
        }

        /// <summary>
        /// Stop stress test
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnStop_Click(object sender, EventArgs e)
        {
            this.IsRunning = false;
        }

        /// <summary>
        /// Thread of logging process.
        /// </summary>
        /// <param name="args"></param>
        private void runStressTest(object args)
        {
            var parameters = args as object[];
            var index = 0;
            var repetition = System.Convert.ToInt32(parameters[index++].ToString());
            var israndcat = System.Convert.ToBoolean(parameters[index++]);
            var category = parameters[index++].ToString();
            var israndsev = System.Convert.ToBoolean(parameters[index++]);
            var severity = parameters[index++].ToString();
            var msg = parameters[index++].ToString();

            var repcount = 1;

            // For category random.
            var catrand = new System.Random();
            var catnames = System.Enum.GetNames(typeof(DefaultCategory));

            // For severity random.
            var sevrand = new System.Random();
            var sevnames = System.Enum.GetNames(typeof(Severity));

            // UI thread dispatcher.
            var progressDispatcher = new System.Action<int, int>(this.progressDispatcher);
            var errorDispatcher = new System.Action<string>(this.errorDispatcher);
            var finishDispatcher = new System.Action(this.finishDispatcher);

            this.IsRunning = true;
            while (this.IsRunning && repcount <= repetition)
            {
                // Add the number of repetitions to the message.
                var message = System.String.Format("Rep.{0}. {1}", repcount, msg);

                // Get call stack.
                var stack = new System.Diagnostics.StackFrame(1);
                var path = stack.GetMethod().ToString();

                // For random.
                if (israndcat) category = catnames[catrand.Next() % catnames.Length];
                if (israndsev) severity = sevnames[sevrand.Next() % sevnames.Length];

                // Call LoggingFoot API
                var result = FtsdkCSWrapper.FTCORE_SendMessage(path, category, severity, message);

                // Error occurred
                if (result == FtsdkCSWrapper.FTCORE_RESULT.FTCORE_SUCCESS)
                {
                    // progress update to UI
                    this.Invoke(progressDispatcher, repcount, repetition);
                    repcount++;

                    // Interval
                    if(this.RepetitionInterval > 0) System.Threading.Thread.CurrentThread.Join(this.RepetitionInterval);
                }
                else
                {
                    var error = System.String.Format("Error occurred. Result: {0}", result.ToString());
                    this.Invoke(errorDispatcher, error);
                    this.IsRunning = false;
                    break;
                }
            }

            // Finish
            this.Invoke(finishDispatcher);
        }

        /// <summary>
        /// Update progress to UI.
        /// </summary>
        /// <param name="repcount">current count</param>
        /// <param name="repetition">maximum count</param>
        private void progressDispatcher(int repcount, int repetition)
        {
            var progress = (int)(((double)repcount / repetition) * 100.0);
            this.prgProgress.Value = progress;
            this.lblProgress.Text = System.String.Format("{0}%", progress);
        }

        /// <summary>
        /// Update state to UI.
        /// </summary>
        /// <param name="error">error message</param>
        private void errorDispatcher(string error)
        {
            this.setIdleState();
            System.Windows.Forms.MessageBox.Show(error);
        }

        /// <summary>
        /// Thread of logging process finished.
        /// </summary>
        private void finishDispatcher()
        {
            this.setIdleState();
        }
        #endregion Stress test
    }
}
