#-----------------------------------------------------------------------------------------------------------------------------
# Python wrapper for the LoggingFoot FTSDK.
#
# Copyright © 2025 Quantyworks Software
#-----------------------------------------------------------------------------------------------------------------------------
from ctypes import windll, Structure, c_int, c_uint32, c_ushort, c_char_p
from enum import IntEnum, Enum

_DLL_NAME = "..\\ftsdk\\bin\\ft_cli.dll"
"""LoggingFoot client module name."""

# Load the DLL
ftcore = windll.LoadLibrary(_DLL_NAME)
"""LoggingFoot client module object."""

class FtsdkPYWrapper:
    """
    LoggingFoot FTSDK Core published APIs

    """

    #region Enums
    class FTCORE_RESULT(IntEnum):
        """
        Represents the execution result of the LoggingFoot client.
        """

        FTCORE_UNKNOWN_STATE           = -1 # The state is undefined.
        FTCORE_SUCCESS                 = 0  # Operation completed successfully.


        #-- Events --#

        FTCORE_EVT_CLIENT_DETECTDISCON = 4  # Client detected disconnection from the server.
        FTCORE_EVT_CLIENT_RECEIVED     = 5	# Client received data from the server.


        #-- Errors --#

        FTCORE_ERR_CLIENT_NOTSERVER    = 21 # The Server is not running or does not exist.
        FTCORE_ERR_CLIENT_FAILSERVER   = 22 # Failed to start the logging server.

        FTCORE_ERR_CLIENT_NOEXIST      = 23 # Called before the process was created.
        FTCORE_ERR_CLIENT_ALREADY      = 24 # The process has already started.
        FTCORE_ERR_CLIENT_PARAMETER    = 25 # Invalid parameters were provided.
        FTCORE_ERR_CLIENT_HOSTINFO     = 26 # Failed to retrieve host information.
        FTCORE_ERR_CLIENT_SOCKET       = 27 # Failed to create socket.
        FTCORE_ERR_CLIENT_REFUSED      = 28 # The target process is not accepting connections or refused the connection.
        FTCORE_ERR_CLIENT_UNREACHED    = 29 # The network path to the destination is unknown.
        FTCORE_ERR_CLIENT_CONNECT      = 30 # Connection failed due to an unknown cause.
        FTCORE_ERR_CLIENT_ESTABLISH    = 31 # An error occurred during the communication establishment process.
        FTCORE_ERR_CLIENT_MESSAGESEND  = 32 # Failed to send message.
        FTCORE_ERR_CLIENT_MESSAGERECV  = 33 # Failed to receive message.
        FTCORE_ERR_CLIENT_CONNUNKNOWN  = 34 # Received a notification accepting a connection from an unknown server.
    #endregion

    #region FTCORE External API Definitions
    class FTCORE_APIs:
        """
        FTCORE External API Definitions
        """

        @staticmethod
        def FTCORE_StartProcess(servername: str, portnumber: int) -> int:
            """
            Starts the LoggingFoot client process.

            Args:
                servername : The server name or IP address of the destination to connect to.
                portnumber : The port number of the destination server (49152–65535).

            Returns:
                FTCORE_RESULT code.
            """
            func = ftcore.FTCORE_StartProcess
            func.argtypes = [c_char_p, c_ushort]
            func.restype = c_uint32
            return func(servername.encode(), portnumber)


        @staticmethod
        def FTCORE_StartTriggerWithParam(serverpath: str, servername: str, portnumber: int, viewstate: int, topmost: int) -> int:
            """
            After the server process is started with the parameters passed, the client process is started.

            Args:
                serverpath  : The full path to the destination server executable.
                servername  : The server name or IP address to connect to.
                portnumber  : The port number of the destination server (49152–65535).
                viewstate   : The window state of the server process.
                                0: Default
                                1: Maximized
                                2: Minimized to the taskbar
                                3: Hidden in the system tray
                                4: Fully hidden
                topmost     : Topmost flag (0 or 1).
                                0: Disabled
                                1: Enabled

            Returns:
                FTCORE_RESULT code.
            """
            func = ftcore.FTCORE_StartTriggerWithParam
            func.argtypes = [c_char_p, c_char_p, c_ushort, c_int, c_int]
            func.restype = c_uint32
            return func(serverpath.encode(), servername.encode(), portnumber, viewstate, topmost)


        @staticmethod
        def FTCORE_StartTriggerWithSetup(serverpath: str, servername: str, portnumber: int) -> int:
            """
            Starts the client process after the server process has been launched, using the setup parameters.

            Args:
                serverpath  : The full path to the destination server executable.
                servername  : The server name or IP address to connect to.
                portnumber  : The port number of the destination server (49152–65535).

            Returns:
                FTCORE_RESULT code.
            """
            func = ftcore.FTCORE_StartTriggerWithSetup
            func.argtypes = [c_char_p, c_char_p, c_ushort]
            func.restype = c_uint32
            return func(serverpath.encode(), servername.encode(), portnumber)

        @staticmethod
        def FTCORE_ExitProcess() -> int:
            """
            Terminates the client process.
            This function must be called to terminate the process if it was started using FTCORE_StartProcess().

            Returns:
                FTCORE_RESULT code.
            """
            func = ftcore.FTCORE_ExitProcess
            func.restype = c_uint32
            return func()

        @staticmethod
        def FTCORE_ExitTrigger() -> int:
            """
            Terminates the client process after the server process has exited.
            This function must be called to terminate the process if it was started using either FTCORE_StartTriggerWithParam() or FTCORE_StartTriggerWithSetup().
            If no server process is running, only the client process will be terminated.

            Returns:
                FTCORE_RESULT code.
            """
            func = ftcore.FTCORE_ExitTrigger
            func.restype = c_uint32
            return func()

        @staticmethod
        def FTCORE_SendMessage(path: str, category: str, severity: str, message: str) -> int:
            """
            Sends a message to the server process.

            Args:
                path        : The caller's path or source identifier.
                category    : The log category.
                severity    : The log severity level.
                message     : The message content to send.

            Returns:
                FTCORE_RESULT code.
            """
            func = ftcore.FTCORE_SendMessage
            func.argtypes = [c_char_p, c_char_p, c_char_p, c_char_p]
            func.restype = c_uint32
            return func(path.encode(), category.encode(), severity.encode(), message.encode())

        @staticmethod
        def FTCORE_SetWindowState(state: int) -> int:
            """
            Instructs the server process to change its window display state.

            Args:
                state       : The desired window state of the server process.
                                0: Default
                                1: Maximized
                                2: Minimized to the taskbar
                                3: Hidden in the system tray
                                4: Fully hidden

            Returns:
                FTCORE_RESULT code.
            """
            func = ftcore.FTCORE_SetWindowState
            func.argtypes = [c_int]
            func.restype = c_uint32
            return func(state)

        @staticmethod
        def FTCORE_LoadDefaultCategory() -> int:
            """
            Instructs the server process to load the default logging category values.

            Returns:
                FTCORE_RESULT code.
            """
            func = ftcore.FTCORE_LoadDefaultCategory
            func.restype = c_uint32
            return func()

        @staticmethod
        def FTCORE_SetCustomCategory(categories: str) -> int:
            """
            Instructs the server process to apply custom logging categories.

            Args:
                categories  : One or more category names, separated by commas if multiple.

            Returns:
                FTCORE_RESULT code.
            """
            func = ftcore.FTCORE_SetCustomCategory
            func.argtypes = [c_char_p]
            func.restype = c_uint32
            return func(categories.encode())

        @staticmethod
        def FTCORE_SetCustomCategoryFromFile(filename: str) -> int:
            """
            Instructs the server process to apply the loaded category settings from the specified file.

            Args:
                filename    : Full path of category setting file name.

            Returns:
                FTCORE_RESULT code.
            """
            func = ftcore.FTCORE_SetCustomCategoryFromFile
            func.argtypes = [c_char_p]
            func.restype = c_uint32
            return func(filename.encode())
    #endregion

    #region LoggingClient Wrapper
    @staticmethod
    def FTCORE_StartProcess(servername: str, portnumber: int) -> FTCORE_RESULT:
        """
        Starts the LoggingFoot client process.

        Args:
            servername  : The server name or IP address of the destination to connect to.
            portnumber  : The port number of the destination server (49152–65535).

        Returns:
            SUCCESS     : FTCORE_SUCCESS.
            FAIL        : FTCORE_ERR_CLIENT_ALREADY
                        : FTCORE_ERR_CLIENT_SOCKET
                        : FTCORE_ERR_CLIENT_REFUSED
                        : FTCORE_ERR_CLIENT_UNREACHED
                        : FTCORE_ERR_CLIENT_CONNECT
                        : FTCORE_ERR_CLIENT_ESTABLISH
        """
        result = FtsdkPYWrapper.FTCORE_APIs.FTCORE_StartProcess(servername, portnumber)
        return FtsdkPYWrapper.FTCORE_RESULT(result)


    @staticmethod
    def FTCORE_StartTriggerWithParam(serverpath: str, servername: str, portnumber: int, viewstate: int, topmost: int) -> FTCORE_RESULT:
        """
        After the server process is started with the parameters passed, the client process is started.

        Args:
            serverpath  : The full path to the destination server executable.
            servername  : The server name or IP address to connect to.
            portnumber  : The port number of the destination server (49152–65535).
            viewstate   : The window state of the server process.
                            0: Default
                            1: Maximized
                            2: Minimized to the taskbar
                            3: Hidden in the system tray
                            4: Fully hidden
            topmost     : Topmost flag (0 or 1).
                            0: Disabled
                            1: Enabled

        Returns:
            SUCCESS     : FTCORE_SUCCESS.
            FAIL        : FTCORE_ERR_CLIENT_NOTSERVER
                        : FTCORE_ERR_CLIENT_FAILSERVER
                        : FTCORE_ERR_CLIENT_ALREADY
                        : FTCORE_ERR_CLIENT_SOCKET
                        : FTCORE_ERR_CLIENT_REFUSED
                        : FTCORE_ERR_CLIENT_UNREACHED
                        : FTCORE_ERR_CLIENT_CONNECT
                        : FTCORE_ERR_CLIENT_ESTABLISH
        """ 
        result = FtsdkPYWrapper.FTCORE_APIs.FTCORE_StartTriggerWithParam(serverpath, servername, portnumber, viewstate, topmost)
        return FtsdkPYWrapper.FTCORE_RESULT(result)


    @staticmethod
    def FTCORE_StartTriggerWithSetup(serverpath: str, servername: str, portnumber: int) -> FTCORE_RESULT:
        """
        Starts the client process after the server process has been launched, using the setup parameters.

        Args:
            serverpath  : The full path to the destination server executable.
            servername  : The server name or IP address to connect to.
            portnumber  : The port number of the destination server (49152–65535).

        Returns:
            SUCCESS     : FTCORE_SUCCESS.
            FAIL        : FTCORE_ERR_CLIENT_NOTSERVER
                        : FTCORE_ERR_CLIENT_FAILSERVER
                        : FTCORE_ERR_CLIENT_ALREADY
                        : FTCORE_ERR_CLIENT_SOCKET
                        : FTCORE_ERR_CLIENT_REFUSED
                        : FTCORE_ERR_CLIENT_UNREACHED
                        : FTCORE_ERR_CLIENT_CONNECT
                        : FTCORE_ERR_CLIENT_ESTABLISH
        """
        result = FtsdkPYWrapper.FTCORE_APIs.FTCORE_StartTriggerWithSetup(serverpath, servername, portnumber)
        return FtsdkPYWrapper.FTCORE_RESULT(result)


    @staticmethod
    def FTCORE_ExitProcess() -> FTCORE_RESULT:
        """
        Terminates the client process.
        This function must be called to terminate the process if it was started using FTCORE_StartProcess().

        Returns:
            SUCCESS     : FTCORE_SUCCESS.
            FAIL        : -----
        """
        result = FtsdkPYWrapper.FTCORE_APIs.FTCORE_ExitProcess()
        return FtsdkPYWrapper.FTCORE_RESULT(result)


    @staticmethod
    def FTCORE_ExitTrigger() -> FTCORE_RESULT:
        """
        Terminates the client process after the server process has exited.
        This function must be called to terminate the process if it was started using either FTCORE_StartTriggerWithParam() or FTCORE_StartTriggerWithSetup().
        If no server process is running, only the client process will be terminated.

        Returns:
            SUCCESS     : FTCORE_SUCCESS.
            FAIL        : -----
        """
        result = FtsdkPYWrapper.FTCORE_APIs.FTCORE_ExitTrigger()
        return FtsdkPYWrapper.FTCORE_RESULT(result)


    @staticmethod
    def FTCORE_SendMessage(path: str, category: str, severity: str, message: str) -> FTCORE_RESULT:
        """
        Sends a message to the server process.

        Args:
            path        : The caller's path or source identifier.
            category    : The log category.
            severity    : The log severity level.
            message     : The message content to send.

        Returns:
            SUCCESS     : FTCORE_SUCCESS.
            FAIL        : FTCORE_ERR_CLIENT_NOEXIST
                        : FTCORE_ERR_CLIENT_PARAMETER
                        : FTCORE_ERR_CLIENT_MESSAGESEND
        """
        result = FtsdkPYWrapper.FTCORE_APIs.FTCORE_SendMessage(path, category, severity, message)
        return FtsdkPYWrapper.FTCORE_RESULT(result)


    @staticmethod
    def FTCORE_SetWindowState(state: int) -> FTCORE_RESULT:
        """
        Instructs the server process to change its window display state.

        Args:
            state       : The desired window state of the server process.
                            0: Default
                            1: Maximized
                            2: Minimized to the taskbar
                            3: Hidden in the system tray
                            4: Fully hidden

        Returns:
            SUCCESS     : FTCORE_SUCCESS.
            FAIL        : FTCORE_ERR_CLIENT_NOEXIST
                        : FTCORE_ERR_CLIENT_PARAMETER
                        : FTCORE_ERR_CLIENT_MESSAGESEND
        """
        result = FtsdkPYWrapper.FTCORE_APIs.FTCORE_SetWindowState(state)
        return FtsdkPYWrapper.FTCORE_RESULT(result)


    @staticmethod
    def FTCORE_LoadDefaultCategory() -> FTCORE_RESULT:
        """
        Instructs the server process to load the default logging category values.

        Returns:
            SUCCESS     : FTCORE_SUCCESS.
            FAIL        : FTCORE_ERR_CLIENT_NOEXIST
                        : FTCORE_ERR_CLIENT_PARAMETER
                        : FTCORE_ERR_CLIENT_MESSAGESEND
        """
        result = FtsdkPYWrapper.FTCORE_APIs.FTCORE_LoadDefaultCategory()
        return FtsdkPYWrapper.FTCORE_RESULT(result)


    @staticmethod
    def FTCORE_SetCustomCategory(categories: str) -> FTCORE_RESULT:
        """
        Instructs the server process to apply custom logging categories.

        Args:
            categories  : One or more category names, separated by commas if multiple.

        Returns:
            SUCCESS     : FTCORE_SUCCESS.
            FAIL        : FTCORE_ERR_CLIENT_NOEXIST
                        : FTCORE_ERR_CLIENT_PARAMETER
                        : FTCORE_ERR_CLIENT_MESSAGESEND
        """
        result = FtsdkPYWrapper.FTCORE_APIs.FTCORE_SetCustomCategory(categories)
        return FtsdkPYWrapper.FTCORE_RESULT(result)

    @staticmethod
    def FTCORE_SetCustomCategoryFromFile(filename: str) -> FTCORE_RESULT:
        """
        Instructs the server process to apply the loaded category settings from the specified file.

        Args:
            filename    : Full path of category setting file name.

        Returns:
            SUCCESS     : FTCORE_SUCCESS.
            FAIL        : FTCORE_ERR_CLIENT_NOEXIST
                        : FTCORE_ERR_CLIENT_PARAMETER
                        : FTCORE_ERR_CLIENT_MESSAGESEND
        """
        result = FtsdkPYWrapper.FTCORE_APIs.FTCORE_SetCustomCategoryFromFile(filename)
        return FtsdkPYWrapper.FTCORE_RESULT(result)
    #endregion