import win32api     # need to install pywin32
import os
import sys
import msvcrt
import random
import inspect
from pathlib import Path
from enum import Enum

import FtsdkPYWrapper as ftsdk

class DefaultCategory(Enum):
    """Default category list."""

    NON     = 0     # No categorized
    APP     = 1     # Application
    SYSTEM  = 2     # System
    USER    = 3     # User operation
    UI      = 4     # GUI operation
    WF      = 5     # Work flow
    DEVICE  = 6     # Device
    DEBUG   = 7     # Debug
    STEP    = 8     # Step
    EVENT   = 9     # Event
    COMM    = 10    # Communication port

class Severity(Enum):
    """Logging severity list."""

    NON     = 0     # No setting
    INFO    = 1     # Information
    NOTICE  = 2     # Notice
    WARNING = 3     # Warning
    ERROR   = 4     # Normal error
    FATAL   = 5     # Fatal error

def getSDKVersion() -> str:
    """
    Get Logging Foot SDK version. 

    Returns: FTSDK version string.
    """

    version = "-----"
    try:
        # Get the path of the DLL
        pathobj = Path(__file__).parent.parent.resolve()
        dllpath = os.path.join(pathobj, "ftsdk", "bin", "ft_cli.dll")
        
        # Get the version information
        verinfo = win32api.GetFileVersionInfo(dllpath, "\\")
        ms = verinfo['FileVersionMS']
        ls = verinfo['FileVersionLS']
        version = f"{ms >> 16}.{ms & 0xFFFF}.{ls >> 16}.{ls & 0xFFFF}"

    except Exception as e:
        print(f"Error getting SDK version: {e}")
    return version

def OutputLog(message: str):
    """
    Outputs a log message.

    Args:
        message : The content of the log message to be sent.
    """
    # Random number generator
    rand = random.Random()

    # Get caller method
    stack = inspect.stack()
    if len(stack) > 1:
        path = stack[1].function
    else:
        path = "<unknown>"

    # Get log categories at random
    categories = [e.name for e in DefaultCategory]
    category = rand.choice(categories)

    # Get log severity at random
    severities = [e.name for e in Severity]
    severity = rand.choice(severities)

    # API execution
    print("> Called FTCORE_SendMessage(...)")
    result = ftsdk.FtsdkPYWrapper.FTCORE_SendMessage(path, category, severity, message)
    print(f"> result: {result.name}")


def main():
    """ LoggingFoot sample program. """
    """ How to use FTCORE_StartProcess and FTCORE_SetDefaultCategory. """

    copyright = "Copyright (C) 2025 Quantyworks Software"
    version = getSDKVersion()
  
    print("**********************************************************************")
    print("")
    print("  LoggingFoot FTSDK sample program.")
    print("  How to use FTCORE_StartProcess and FTCORE_LoadDefaultCategory.")
    print("")
    print(f"  {copyright}")
    print(f"                                                  FTSDK ver.{version}")
    print("**********************************************************************")
    print("")
    print("First, Please start the Logging server.")
    print("And next, press any key. (Press Esc key to exit this program.)")
    print("Logging client will start.")
    print("")

    # Destination server information.
    server = "localhost"
    port = 50500

    # API execution result.
    result = ftsdk.FtsdkPYWrapper.FTCORE_RESULT.FTCORE_UNKNOWN_STATE

    # Wait for key input.
    print("> Wait for any key to be pressed ...")
    print("> ", end="", flush=True)
    key = msvcrt.getch().decode()
    print(f"{key}")
    print(f"> {key} key was pressed.")
    if key == '\x1b':   # ESC key

        #- Exit program -#
        pass

    else:

        #- Startup Logging Client -//
        print("> Called FTCORE_StartProcess()..");
        result = ftsdk.FtsdkPYWrapper.FTCORE_StartProcess(server, port)
        print(f"> result: {result.name}")
        print("")

    if result == ftsdk.FtsdkPYWrapper.FTCORE_RESULT.FTCORE_SUCCESS:

        #- Directs the loading of the default category stored on the Logging Server. -#
        
        # Directs loading of the default category.
        print("> Called FTCORE_LoadDefaultCategory()..")
        result = ftsdk.FtsdkPYWrapper.FTCORE_LoadDefaultCategory()
        print(f"> result: {result.name}")
        print("")

    if result == ftsdk.FtsdkPYWrapper.FTCORE_RESULT.FTCORE_SUCCESS:

        #- Log out inpout keys -#
        
        print("> Press any key then, Output log message.")
        print("")

        print("> ", end="", flush=True)
        key = msvcrt.getch().decode()
        while key != '\x1b':   # ESC key
            print(f"{key}")
            print(f"> {key} key was pressed.")
            print("")
            logmsg = f"{key} key was pressed."
            OutputLog(logmsg)
            key = msvcrt.getch().decode()

        #- Exit program -#

        if key == '\x1b':   # ESC key

            # Logging Client exit process
            print("> Called FTCORE_ExitProcess()..")
            result = ftsdk.FtsdkPYWrapper.FTCORE_ExitProcess()

            print(f"> result: {result.name}")
            print("")

    # Press any key to exit program.
    print("Press any key again to exit program")
    msvcrt.getch()

if __name__ == "__main__":
    main()