//----------------------------------------------------------------------------------------------
// C++ wrapper for the LoggingFoot FTSDK.
//
// Copyright © 2025 Quantyworks Software
//----------------------------------------------------------------------------------------------
#include <iostream>
#include <string>
#include <sstream>
#include <windows.h>
#include "ftsdk_wrapper.h"
#include "ftsdk_core.h"
#pragma comment(lib, "ft_cli.lib")

#define CATEGORY_LEN 32
#define SEVERITY_LEN 32
#define MAX_RECV_SIZE 4097

//-----------------------------------------------------------------------------------------------------------------------------
// Logging Foot FTSDK C++ Wrapper Layer.
//
// Usage of FTCORE_StartProcess().
//
//
// servername		(IN)		: The server name or IP address of the destination to connect to.
// portnumber		(IN)		: The port number of the destination server(49152 ~ 65535).
//
// return			(SUCCESS)	: FTCORE_SUCCESS                (0)
//					(ONERROR)	: FTCORE_ERR_CLIENT_ALREADY     (24)
//								: FTCORE_ERR_CLIENT_SOCKET      (27)
//								: FTCORE_ERR_CLIENT_REFUSED     (28)
//								: FTCORE_ERR_CLIENT_UNREACHED   (29)
//								: FTCORE_ERR_CLIENT_CONNECT     (30)
//								: FTCORE_ERR_CLIENT_ESTABLISH   (31)
//-----------------------------------------------------------------------------------------------------------------------------
int FTSDKWrapper::ftcore_StartProcess(std::string servername, unsigned short portnumber)
{
    char server[MAX_PATH] = {}; ::sprintf_s(server, MAX_PATH, "%s", servername.c_str());
    
    // Call FTSDK API
    return FTCORE_StartProcess(server, portnumber);
}

//-----------------------------------------------------------------------------------------------------------------------------
// Logging Foot FTSDK C++ Wrapper Layer.
//
// Usage of FTCORE_StartTriggerWithParam().
//
//
// serverpath		(IN)		: The full path to the destination server executable.
// servername		(IN)		: The server name or IP address to connect to.
// portnumber		(IN)		: The port number of the destination server (49152–65535).
// viewstate		(IN)		: The window state of the server process.
//								: (0: Default window size)
//								: (1: Maximized window)
//								: (2: Minimized in the taskbar)
//								: (3: Hidden in the system tray)
//								: (4: Fully hidden)
// topmost			(IN)		: Specifies whether the window should stay on top of other windows.
//
// return			(SUCCESS)	: FTCORE_SUCCESS                (0)
//					(ONERROR)	: FTCORE_ERR_CLIENT_NOTSERVER   (21)
//								: FTCORE_ERR_CLIENT_FAILSERVER  (22)
//								: FTCORE_ERR_CLIENT_ALREADY     (24)
//								: FTCORE_ERR_CLIENT_SOCKET      (27)
//								: FTCORE_ERR_CLIENT_REFUSED     (28)
//								: FTCORE_ERR_CLIENT_UNREACHED   (29)
//								: FTCORE_ERR_CLIENT_CONNECT     (30)
//								: FTCORE_ERR_CLIENT_ESTABLISH   (31)
//-----------------------------------------------------------------------------------------------------------------------------
int FTSDKWrapper::ftcore_StartTriggerWithParam(std::string serverpath, std::string servername, unsigned short portnumber, int viewstate, int topmost)
{
    char path[MAX_PATH] = {}; ::sprintf_s(path, MAX_PATH, "%s", serverpath.c_str());
    char name[MAX_PATH] = {}; ::sprintf_s(name, MAX_PATH, "%s", servername.c_str());

    // Call FTSDK API
    return FTCORE_StartTriggerWithParam(path, name, portnumber, viewstate, topmost);
}

//-----------------------------------------------------------------------------------------------------------------------------
// Logging Foot FTSDK C++ Wrapper Layer.
//
// Usage of FTCORE_StartTriggerWithSetup().
//
//
// serverpath		(IN)		: The full path to the destination server executable.
// servername		(IN)		: The server name or IP address to connect to.
// portnumber		(IN)		: The port number of the destination server (49152–65535).
//
// return			(SUCCESS)	: FTCORE_SUCCESS                (0)
//					(ONERROR)	: FTCORE_ERR_CLIENT_NOTSERVER   (21)
//								: FTCORE_ERR_CLIENT_FAILSERVER  (22)
//								: FTCORE_ERR_CLIENT_ALREADY     (24)
//								: FTCORE_ERR_CLIENT_SOCKET      (27)
//								: FTCORE_ERR_CLIENT_REFUSED     (28)
//								: FTCORE_ERR_CLIENT_UNREACHED   (29)
//								: FTCORE_ERR_CLIENT_CONNECT     (30)
//								: FTCORE_ERR_CLIENT_ESTABLISH   (31)
//-----------------------------------------------------------------------------------------------------------------------------
int FTSDKWrapper::ftcore_StartTriggerWithSetup(std::string serverpath, std::string servername, unsigned short portnumber)
{
    char path[MAX_PATH] = {}; ::sprintf_s(path, MAX_PATH, "%s", serverpath.c_str());
    char name[MAX_PATH] = {}; ::sprintf_s(name, MAX_PATH, "%s", servername.c_str());

    // Call FTSDK API
    return FTCORE_StartTriggerWithSetup(path, name, portnumber);
}

//-----------------------------------------------------------------------------------------------------------------------------
// Logging Foot FTSDK C++ Wrapper Layer.
//
// Usage of FTCORE_ExitProcess().
//
//
// return			(SUCCESS)	: FTCORE_SUCCESS
//					(ONERROR)	: --------
//-----------------------------------------------------------------------------------------------------------------------------
int FTSDKWrapper::ftcore_ExitProcess()
{
    // Call FTSDK API
    return FTCORE_ExitProcess();
}

//-----------------------------------------------------------------------------------------------------------------------------
// Logging Foot FTSDK C++ Wrapper Layer.
//
// Usage of FTCORE_ExitTrigger().
//
//
// return			(SUCCESS)	: FTCORE_SUCCESS
//					(ONERROR)	: --------
//-----------------------------------------------------------------------------------------------------------------------------
int FTSDKWrapper::ftcore_ExitTrigger()
{
    // Call FTSDK API
    return FTCORE_ExitTrigger();
}

//-----------------------------------------------------------------------------------------------------------------------------
// Logging Foot FTSDK C++ Wrapper Layer.
//
// Usage of FTCORE_SendMessage().
//
//
// path				(IN)		: The caller's path or source identifier.
// category			(IN)		: The log category.
// severity			(IN)		: The log severity level.
// message			(IN)		: The message content to send.
//
// return			(SUCCESS)	: FTCORE_SUCCESS                (0)
//					(ONERROR)	: FTCORE_ERR_CLIENT_NOEXIST     (23)
//								: FTCORE_ERR_CLIENT_PARAMETER   (25)
//								: FTCORE_ERR_CLIENT_MESSAGESEND (32)
//-----------------------------------------------------------------------------------------------------------------------------
int FTSDKWrapper::ftcore_SendMessage(std::string path, std::string category, std::string severity, std::string message)
{
    char caller[MAX_PATH] = {}; ::sprintf_s(caller, MAX_PATH, "%s", path.c_str());
    char cat[CATEGORY_LEN] = {}; ::sprintf_s(cat, CATEGORY_LEN, "%s", category.c_str());
    char sev[SEVERITY_LEN] = {}; ::sprintf_s(sev, SEVERITY_LEN, "%s", severity.c_str());
    char code[MAX_RECV_SIZE] = {};
    char msg[MAX_RECV_SIZE] = {};

    // Convert csv formatted UTF-8 code point string to number value.
    this->_convertCsvToNum(message, ',', code);

    // Revert UTF-8 code point to string.
    this->_converStringUtf8(code, MAX_RECV_SIZE, msg);

    // Call FTSDK API
    return FTCORE_SendMessage(caller, cat, sev, msg);
}

//-----------------------------------------------------------------------------------------------------------------------------
// Logging Foot FTSDK C++ Wrapper Layer.
//
// Usage of FTCORE_SetWindowState().
//
//
// state			(IN)		: 0 SHOW_WINDOW_NORMALY		Default window size
//								: 1 SHOW_WINDOW_MAXIMIZE	Maximized window
//								: 2 MINIMIZE_IN_TASKBAR		Minimized to the taskbar
//								: 3 HIDE_IN_TASKTRAY		Hidden in the system tray
//   							: 4 WINDOW_HIDDEN			Fully hidden
//
// return			(SUCCESS)	: FTCORE_SUCCESS                (0)
//					(ONERROR)	: FTCORE_ERR_CLIENT_NOEXIST     (23)
//								: FTCORE_ERR_CLIENT_PARAMETER   (25)
//								: FTCORE_ERR_CLIENT_MESSAGESEND (32)
//-----------------------------------------------------------------------------------------------------------------------------
int FTSDKWrapper::ftcore_SetWindowState(int state)
{
    // Call FTSDK API
    return FTCORE_SetWindowState(state);
}

//-----------------------------------------------------------------------------------------------------------------------------
// Logging Foot FTSDK C++ Wrapper Layer.
//
// Usage of FTCORE_LoadDefaultCategory().
//
//
// return			(SUCCESS)	: FTCORE_SUCCESS                (0)
//					(ONERROR)	: FTCORE_ERR_CLIENT_NOEXIST     (23)
//								: FTCORE_ERR_CLIENT_PARAMETER   (25)
//								: FTCORE_ERR_CLIENT_MESSAGESEND (32)
//-----------------------------------------------------------------------------------------------------------------------------
int FTSDKWrapper::ftcore_LoadDefaultCategory()
{
    // Call FTSDK API
    return FTCORE_LoadDefaultCategory();
}

//-----------------------------------------------------------------------------------------------------------------------------
// Logging Foot FTSDK C++ Wrapper Layer.
//
// Usage of FTCORE_SetCustomCategory().
//
//
// categories		(IN)		: One or more category names, separated by commas if multiple.
//
// return			(SUCCESS)	: FTCORE_SUCCESS                (0)
//					(ONERROR)	: FTCORE_ERR_CLIENT_NOEXIST     (23)
//								: FTCORE_ERR_CLIENT_PARAMETER   (25)
//								: FTCORE_ERR_CLIENT_MESSAGESEND (32)
//-----------------------------------------------------------------------------------------------------------------------------
int FTSDKWrapper::ftcore_SetCustomCategory(std::string categories)
{
    char cat[CATEGORY_LEN] = {}; ::sprintf_s(cat, CATEGORY_LEN, "%s", categories.c_str());

    // Call FTSDK API
    return FTCORE_SetCustomCategory(cat);
}

//-----------------------------------------------------------------------------------------------------------------------------
// Logging Foot FTSDK C++ Wrapper Layer.
//
// Usage of FTCORE_SetCustomCategoryFromFile().
//
//
// filename 		(IN)		: Full path of category setting file name.
//
// return			(SUCCESS)	: FTCORE_SUCCESS                (0)
//					(ONERROR)	: FTCORE_ERR_CLIENT_NOEXIST     (23)
//								: FTCORE_ERR_CLIENT_PARAMETER   (25)
//								: FTCORE_ERR_CLIENT_MESSAGESEND (32)
//-----------------------------------------------------------------------------------------------------------------------------
int FTSDKWrapper::ftcore_SetCustomCategoryFromFile(std::string filename)
{
     char file[MAX_PATH] = {}; ::sprintf_s(file, MAX_PATH, "%s", filename.c_str());

    // Call FTSDK API
    return FTCORE_SetCustomCategoryFromFile(file);
}