﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using FTSDK_Wrapper;

namespace Sample_StartupServer
{
    public partial class frmMain : Form
    {
        #region Definitions
        /// <summary>
        /// Default category list.
        /// </summary>
        enum DefaultCategory
        {
            /// <summary>
            /// No categorized
            /// </summary>
            NON,
            /// <summary>
            /// Application
            /// </summary>
            APP,
            /// <summary>
            /// System
            /// </summary>
            SYSTEM,
            /// <summary>
            /// User operation
            /// </summary>
            USER,
            /// <summary>
            /// GUI operation
            /// </summary>
            UI,
            /// <summary>
            /// Work flow
            /// </summary>
            WF,
            /// <summary>
            /// Device
            /// </summary>
            DEVICE,
            /// <summary>
            /// Debug
            /// </summary>
            DEBUG,
            /// <summary>
            /// Step
            /// </summary>
            STEP,
            /// <summary>
            /// Event
            /// </summary>
            EVENT,
            /// <summary>
            /// Communication port
            /// </summary>
            COMM,
        }

        /// <summary>
        /// Logging severity list.
        /// </summary>
        enum Severity
        {
            /// <summary>
            /// No setting
            /// </summary>
            NON,
            /// <summary>
            /// Information
            /// </summary>
            INFO,
            /// <summary>
            /// Notice
            /// </summary>
            NOTICE,
            /// <summary>
            /// Warning
            /// </summary>
            WARNING,
            /// <summary>
            /// Normal error
            /// </summary>
            ERROR,
            /// <summary>
            /// Fatal error
            /// </summary>
            FATAL,
        }

        /// <summary>
        /// Window state of Logging viewer list.
        /// </summary>
        enum WindowStates
        {
            /// <summary>
            /// Default
            /// </summary>
            SHOW_WINDOW_NORMALY,
            /// <summary>
            /// Maximized
            /// </summary>
            SHOW_WINDOW_MAXIMIZE,
            /// <summary>
            /// Minimized to the taskbar
            /// </summary>
            MINIMIZE_IN_TASKBAR,
            /// <summary>
            /// Hidden in the system tray
            /// </summary>
            HIDE_IN_SYSTEMTRAY,
            /// <summary>
            /// Fully hidden
            /// </summary>
            WINDOW_HIDDEN,
        }

        /// <summary>
        /// Call APIs list.
        /// </summary>
        enum CallAPIs
        {
            /// <summary>
            /// After the server process is started with parameters passed, the client process is started.
            /// </summary>
            FTCORE_StartTriggerWithParam,
            /// <summary>
            /// After the server process is started with parameters set during setup, the client process is started.
            /// </summary>
            FTCORE_StartTriggerWithSetup
        }

        /// <summary>
        /// Server process window topmost state.
        /// </summary>
        enum TopmostState
        {
            /// <summary>
            /// Topmost is false.
            /// </summary>
            FALSE = 0,
            /// <summary>
            /// Topmost is true.
            /// </summary>
            TRUE
        }
        #endregion Definitions

        #region Constructor
        /// <summary>
        /// Constructor
        /// </summary>
        public frmMain()
        {
            InitializeComponent();

            // Server name
            this.txtServerName.Text = "localhost";

            // Portnumber
            this.txtPortNumber.Text = "50500";

            // Server path
            this.txtServerPath.Text = "..\\..\\..\\..\\..\\ftviewer\\ft_viewer.exe";

            // Call API combobox
            var apis = System.Enum.GetNames(typeof(CallAPIs));
            System.Array.ForEach(apis, item => this.cmbCallAPI.Items.Add(item));
            this.cmbCallAPI.SelectedIndex = 0;

            // Window state combobox
            var states = System.Enum.GetNames(typeof(WindowStates));
            System.Array.ForEach(states, item => this.cmbWindowState.Items.Add(item));
            this.cmbWindowState.SelectedIndex = 0;

            // Category combobox
            var categories = System.Enum.GetNames(typeof(DefaultCategory));
            System.Array.ForEach(categories, item => this.cmbCategory.Items.Add(item));
            this.cmbCategory.SelectedIndex = 0;

            // Severity combobox
            var severities = System.Enum.GetNames(typeof(Severity));
            System.Array.ForEach(severities, item => this.cmbSeverity.Items.Add(item));
            this.cmbSeverity.SelectedIndex = 0;

            // Topmpst combobox
            var topmost = System.Enum.GetNames(typeof(TopmostState));
            System.Array.ForEach(topmost, item => this.cmbTopmost.Items.Add(item));
            this.cmbTopmost.SelectedIndex = 0;

            // Server path button
            this.btnServerPath.Click += this.btnServerPath_Click;

            // Start/Exit button
            this.btnStartup.Click += this.btnStartupServer_Click;
            this.btnExit.Click += this.btnExitServer_Click;

            // Send button
            this.btnSend.Click += this.btnSend_Click;

            // Windows Forms Event
            this.Load += this.frmMain_Load;
            this.FormClosed += this.frmMain_Closed;
        }
        #endregion Constructor

        #region Windows Forms Event
        /// <summary>
        /// Form load
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void frmMain_Load(object sender, EventArgs e)
        {
            // Initial state
            this.setInitialState();
        }

        /// <summary>
        /// Disconnect from server and exit logging.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void frmMain_Closed(object sender, FormClosedEventArgs e)
        {
            FtsdkCSWrapper.FTCORE_ExitTrigger();
        }
        #endregion Windows Forms Event

        #region Windows Forms State
        /// <summary>
        /// GUI button initial state.
        /// </summary>
        private void setInitialState()
        {
            this.btnSend.Enabled = false;
            this.btnStartup.Enabled = true;
            this.btnExit.Enabled = false;
            this.pnlServerSettings.Enabled = true;
        }

        /// <summary>
        /// GUI button idle state.
        /// </summary>
        private void setIdleState()
        {
            this.btnSend.Enabled = true;
            this.btnStartup.Enabled = false;
            this.btnExit.Enabled = true;
            this.pnlServerSettings.Enabled = false;
        }
        #endregion Windows Forms State

        #region Server Program Startup Settings
        /// <summary>
        /// Server Name
        /// </summary>
        private string ServerName
        {
            get { return this.txtServerName.Text; }
        }
        /// <summary>
        /// Port Number
        /// </summary>
        private int PortNumber
        {
            get
            {
                var defnum = 50500;
                var number = 0;
                if (!int.TryParse(this.txtPortNumber.Text, out number))
                {
                    number = defnum;
                }

                return number;
            }
        }

        /// <summary>
        /// Server program location path
        /// </summary>
        private string ServerPath
        {
            get { return this.txtServerPath.Text; }
            set { this.txtServerPath.Text = value; }
        }

        /// <summary>
        /// Server program window state
        /// </summary>
        private int WindowViewState
        {
            get { return this.cmbWindowState.SelectedIndex; }
        }

        /// <summary>
        /// Selected API from Call APIs List.
        /// </summary>
        private CallAPIs SelectedAPI
        {
            get
            {
                if (this.cmbCallAPI.SelectedIndex == 0) return CallAPIs.FTCORE_StartTriggerWithParam;
                else return CallAPIs.FTCORE_StartTriggerWithSetup;
            }
        }

        /// <summary>
        /// Select LoggingFootViewer.exe file.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnServerPath_Click(object sender, EventArgs e)
        {
            var fd = new System.Windows.Forms.OpenFileDialog();
            fd.Filter = "Server program|*.exe";

            if (fd.ShowDialog() == DialogResult.OK)
            {
                this.ServerPath = fd.FileName;
            }
        }

        /// <summary>
        /// After the server process is started, the client process is started.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnStartupServer_Click(object sender, EventArgs e)
        {
            var path = this.ServerPath;
            var server = this.ServerName;
            var port = this.PortNumber;
            var state = this.WindowViewState;
            var topmost = this.TopMost;
            var result = FtsdkCSWrapper.FTCORE_RESULT.FTCORE_SUCCESS;

            if (this.SelectedAPI == CallAPIs.FTCORE_StartTriggerWithParam)
                result = FtsdkCSWrapper.FTCORE_StartTriggerWithParam(path, server, port, state, topmost);
            else
                result = FtsdkCSWrapper.FTCORE_StartTriggerWithSetup(path, server, port);

            if (result == FtsdkCSWrapper.FTCORE_RESULT.FTCORE_SUCCESS)
            {
                this.setIdleState();
            }
        }

        /// <summary>
        /// After the server process is exited, the client process is exited.
        /// Started by FTCORE_StartTriggerWithParam() or FTCORE_StartTriggerWithSetup().
        /// If there is no server process to exit, only the client process is exited. 
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnExitServer_Click(object sender, EventArgs e)
        {
            FtsdkCSWrapper.FTCORE_ExitTrigger();
            this.setInitialState();
        }
        #endregion Server Program Startup Settings

        #region Logging Message Output
        /// <summary>
        /// Category random checked state
        /// </summary>
        private bool IsCategoryRandom
        {
            get { return this.chkCategoryRandom.Checked; }
        }
        /// <summary>
        /// Selected Category
        /// </summary>
        private string SelectedCategory
        {
            get
            {
                if (this.IsCategoryRandom)
                {
                    var rand = new System.Random();
                    var names = System.Enum.GetNames(typeof(DefaultCategory));
                    return names[rand.Next() % names.Length];
                }
                else return this.cmbCategory.SelectedItem.ToString();
            }
        }

        /// <summary>
        /// Severity random checked state
        /// </summary>
        private bool IsSeverityRandom
        {
            get { return this.chkSeverityRandom.Checked; }
        }
        /// <summary>
        /// Selected Severity
        /// </summary>
        private string SelectedSeverity
        {
            get
            {
                if (this.IsSeverityRandom)
                {
                    var rand = new System.Random();
                    var names = System.Enum.GetNames(typeof(Severity));
                    return names[rand.Next() % names.Length];
                }
                else return this.cmbSeverity.SelectedItem.ToString();
            }
        }

        /// <summary>
        /// Logging output message
        /// </summary>
        private string Message
        {
            get { return this.txtMessage.Text; }
        }
        /// <summary>
        /// Send logging message.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnSend_Click(object sender, EventArgs e)
        {
            // Get call stack.
            var stack = new System.Diagnostics.StackFrame(1);
            var path = stack.GetMethod().ToString();

            // Category
            var category = this.SelectedCategory;

            // Severity
            var severity = this.SelectedSeverity;

            // Logging message
            var message = this.Message;

            // Call LoggingFoot API
            var result = FtsdkCSWrapper.FTCORE_SendMessage(path, category, severity, message);
        }
        #endregion Logging Message Output
    }
}
