﻿Imports System.Text

Public Class frmMain

#Region "Definitions"
    ''' <summary>
    ''' Maximum message size (in bytes).
    ''' </summary>
    Const MAX_MESSAGE_SIZE As Integer = 4096
    ''' <summary>
    ''' Default category list.
    ''' </summary>
    Enum DefaultCategory
        ''' <summary>
        ''' No categorized
        ''' </summary>
        NON
        ''' <summary>
        ''' Application
        ''' </summary>
        APP
        ''' <summary>
        ''' System
        ''' </summary>
        SYSTEM
        ''' <summary>
        ''' User operation
        ''' </summary>
        USER
        ''' <summary>
        ''' GUI operation
        ''' </summary>
        UI
        ''' <summary>
        ''' Work flow
        ''' </summary>
        WF
        ''' <summary>
        ''' Device
        ''' </summary>
        DEVICE
        ''' <summary>
        ''' Debug
        ''' </summary>
        DEBUG
        ''' <summary>
        ''' Step
        ''' </summary>
        STEPLOG
        ''' <summary>
        ''' Event
        ''' </summary>
        EVENTLOG
        ''' <summary>
        ''' Communication port
        ''' </summary>
        COMM
    End Enum

    ''' <summary>
    ''' Logging severity list.
    ''' </summary>
    Enum Severity
        ''' <summary>
        ''' No setting
        ''' </summary>
        NON
        ''' <summary>
        ''' Information
        ''' </summary>
        INFO
        ''' <summary>
        ''' Notice
        ''' </summary>
        NOTICE
        ''' <summary>
        ''' Warning
        ''' </summary>
        WARNING
        ''' <summary>
        ''' Normal error
        ''' </summary>
        ONERROR
        ''' <summary>
        ''' Fatal error
        ''' </summary>
        FATAL
    End Enum
#End Region 'Definitions

#Region "Constructor"
    ''' <summary>
    ''' Constructor
    ''' </summary>
    Public Sub New()

        InitializeComponent()

        ' Connection Settings
        AddHandler Me.btnConnect.Click, AddressOf Me.btnConnect_Click
        AddHandler Me.btnDiscon.Click, AddressOf Me.btnDiscon_Click

        ' Category combobox
        Dim categories = System.Enum.GetNames(GetType(DefaultCategory))
        System.Array.ForEach(categories, Sub(item) Me.cmbCategory.Items.Add(item))
        Me.cmbCategory.SelectedIndex = 0

        ' Severity combobox
        Dim severities = System.Enum.GetNames(GetType(Severity))
        System.Array.ForEach(severities, Sub(item) Me.cmbSeverity.Items.Add(item))
        Me.cmbSeverity.SelectedIndex = 0

        ' Random checkbox
        AddHandler Me.chkCategoryRandom.Click, AddressOf Me.chkCategoryRandom_Click
        AddHandler Me.chkSeverityRandom.Click, AddressOf Me.chkSeverityRandom_Click

        ' Send button
        AddHandler Me.btnSend.Click, AddressOf Me.btnSend_Click

        ' Start/Stop button
        AddHandler Me.btnStart.Click, AddressOf Me.btnStart_Click
        AddHandler Me.btnStop.Click, AddressOf Me.btnStop_Click
        Me.btnStop.Enabled = False

        ' Logging output initial message
        Me.txtMessage.Text = "This is Stress Test of LoggingFoot."

        ' Windows Forms Event
        AddHandler Me.Load, AddressOf Me.frmMain_Load
        AddHandler Me.FormClosed, AddressOf Me.frmMain_Closed

    End Sub
#End Region 'Constructor

#Region "Windows Forms Event"
    ''' <summary>
    ''' Form load
    ''' </summary>
    ''' <param name="sender"></param>
    ''' <param name="e"></param>
    Private Sub frmMain_Load(ByVal sender As System.Object, ByVal e As System.EventArgs)

        ' Initial state
        Me.setInitialState()

    End Sub

    ''' <summary>
    ''' Disconnect from server and exit logging.
    ''' </summary>
    ''' <param name="sender"></param>
    ''' <param name="e"></param>
    Private Sub frmMain_Closed(ByVal sender As System.Object, ByVal e As System.Windows.Forms.FormClosedEventArgs)

        FtsdkVBWrapper.FTCORE_ExitProcess()

    End Sub

    ''' <summary>
    ''' Message text changed event
    ''' </summary>
    ''' <param name="sender"></param>
    ''' <param name="e"></param>
    Private Sub txtMessage_TextChanged(sender As Object, e As EventArgs) Handles txtMessage.TextChanged

        Dim ctrl As System.Windows.Forms.TextBox = TryCast(sender, System.Windows.Forms.TextBox)
        If ctrl Is Nothing Then Exit Sub

        Dim text As String = ctrl.Text
        Dim bytenum As Integer = Encoding.UTF8.GetByteCount(text)

        If bytenum <= MAX_MESSAGE_SIZE Then
            Me.lblMessageBytes.Text = bytenum.ToString()
        Else
            Dim byteCount As Integer = 0
            Dim charCount As Integer = 0

            For Each c As Char In text
                Dim charByteSize As Integer = Encoding.UTF8.GetByteCount(New Char() {c})
                If byteCount + charByteSize > MAX_MESSAGE_SIZE Then
                    Exit For
                End If

                byteCount += charByteSize
                charCount += 1
            Next

            ctrl.Text = text.Substring(0, charCount)
        End If

    End Sub
#End Region 'Windows Forms Event

#Region "Windows Forms State"
    ''' <summary>
    ''' GUI button initial state.
    ''' </summary>
    Private Sub setInitialState()

        Me.btnSend.Enabled = False
        Me.btnStart.Enabled = False
        Me.btnStop.Enabled = False
        Me.btnConnect.Enabled = True
        Me.btnDiscon.Enabled = False

    End Sub

    ''' <summary>
    ''' GUI button idle state.
    ''' </summary>
    Private Sub setIdleState()

        Me.btnSend.Enabled = True
        Me.btnStart.Enabled = True
        Me.btnStop.Enabled = False
        Me.btnConnect.Enabled = False
        Me.btnDiscon.Enabled = True
        Me.grpConnectionSettings.Enabled = True
        Me.pnlStressTestSettings.Enabled = True
        Me.lblStatus.Text = "- Stopped"

    End Sub

    ''' <summary>
    ''' GUI button processing state.
    ''' </summary>
    Private Sub setProcessingState()

        Me.btnSend.Enabled = True
        Me.btnStart.Enabled = False
        Me.btnStop.Enabled = True
        Me.btnDiscon.Enabled = False
        Me.grpConnectionSettings.Enabled = False
        Me.pnlStressTestSettings.Enabled = False
        Me.lblStatus.Text = "- Running.."

    End Sub
#End Region 'Windows Forms State

#Region "Server Program Connection Setttings"
    ''' <summary>
    ''' Server Name
    ''' </summary>
    Private ReadOnly Property ServerName As System.String

        Get
            Return Me.txtServerName.Text
        End Get

    End Property
    ''' <summary>
    ''' Port Number
    ''' </summary>
    Private ReadOnly Property PortNumber As Integer

        Get
            Dim defnum = 50500
            Dim number = 0
            If Not Integer.TryParse(Me.txtPortNumber.Text, number) Then
                number = defnum
            End If

            Return number
        End Get

    End Property

    ''' <summary>
    ''' Connect to server
    ''' </summary>
    ''' <param name="sender"></param>
    ''' <param name="e"></param>
    Private Sub btnConnect_Click(ByVal sender As System.Object, ByVal e As System.EventArgs)

        Dim server = Me.ServerName
        Dim port = Me.PortNumber
        Dim result = FtsdkVBWrapper.FTCORE_StartProcess(server, port)

        If result = FtsdkVBWrapper.FTCORE_RESULT.FTCORE_SUCCESS Then

            ' Connected button state
            Me.setIdleState()
        Else

            Dim msg = System.String.Format("Error occurred. Code={0}.\nPlease make sure LoggingFoot Server is running.", result)
            System.Windows.Forms.MessageBox.Show(msg, "Error", MessageBoxButtons.OK, MessageBoxIcon.Warning)
        End If

    End Sub
    ''' <summary>
    ''' Disconnect from server
    ''' </summary>
    ''' <param name="sender"></param>
    ''' <param name="e"></param>
    Private Sub btnDiscon_Click(ByVal sender As System.Object, ByVal e As System.EventArgs)

        FtsdkVBWrapper.FTCORE_ExitProcess()

        Me.setInitialState()

    End Sub
#End Region 'Server Program Connection Setttings

#Region "Logging Message Output"
    ''' <summary>
    ''' Category random checked state
    ''' </summary>
    Private ReadOnly Property IsCategoryRandom As String

        Get
            Return Me.chkCategoryRandom.Checked
        End Get

    End Property
    ''' <summary>
    ''' Selected Category
    ''' </summary>
    Private ReadOnly Property SelectedCategory As String

        Get
            If Me.IsCategoryRandom Then
                Dim rand = New System.Random()
                Dim names = System.Enum.GetNames(GetType(DefaultCategory))
                Return names(rand.Next() Mod names.Length)
            Else
                Return Me.cmbCategory.SelectedItem.ToString()
            End If
        End Get

    End Property
    ''' <summary>
    ''' Category Random checkbox
    ''' </summary>
    ''' <param name="sender"></param>
    ''' <param name="e"></param>
    Private Sub chkCategoryRandom_Click(ByVal sender As System.Object, ByVal e As System.EventArgs)

        Dim checkbox = CType(sender, System.Windows.Forms.CheckBox)
        Me.cmbCategory.Enabled = Not checkbox.Checked

    End Sub

    ''' <summary>
    ''' Severity random checked state
    ''' </summary>
    Private ReadOnly Property IsSeverityRandom As String

        Get
            Return Me.chkSeverityRandom.Checked
        End Get

    End Property
    ''' <summary>
    ''' Selected Category
    ''' </summary>
    Private ReadOnly Property SelectedSeverity As String

        Get
            If Me.IsSeverityRandom Then
                Dim rand = New System.Random()
                Dim names = System.Enum.GetNames(GetType(Severity))
                Return names(rand.Next() Mod names.Length)
            Else
                Return Me.cmbSeverity.SelectedItem.ToString()
            End If
        End Get

    End Property
    ''' <summary>
    ''' Severity Random checkbox
    ''' </summary>
    ''' <param name="sender"></param>
    ''' <param name="e"></param>
    Private Sub chkSeverityRandom_Click(ByVal sender As System.Object, ByVal e As System.EventArgs)

        Dim checkbox = CType(sender, System.Windows.Forms.CheckBox)
        Me.cmbSeverity.Enabled = Not checkbox.Checked

    End Sub

    ''' <summary>
    ''' Logging output message
    ''' </summary>
    Private ReadOnly Property Message As String

        Get
            Return Me.txtMessage.Text
        End Get

    End Property
    ''' <summary>
    ''' Send logging message.
    ''' </summary>
    ''' <param name="sender"></param>
    ''' <param name="e"></param>
    Private Sub btnSend_Click(ByVal sender As System.Object, ByVal e As System.EventArgs)

        ' Get call stack.
        Dim stack = New System.Diagnostics.StackFrame(1)
        Dim path = stack.GetMethod().ToString()

        ' Category
        Dim category = Me.SelectedCategory

        ' Severity
        Dim severity = Me.SelectedSeverity

        ' Logging message
        Dim message = Me.Message

        ' Call LoggingFoot API
        Dim result = FtsdkVBWrapper.FTCORE_SendMessage(path, category, severity, message)

    End Sub
#End Region 'Logging Message Output

#Region "Stress test"
    ''' <summary>
    ''' Repetition Count
    ''' </summary>
    Private ReadOnly Property RepetitionCount As Integer
        Get
            Return Me.numRepetition.Value
        End Get
    End Property
    ''' <summary>
    ''' Repetition Interval
    ''' </summary>
    Private ReadOnly Property RepetitionInterval As Integer
        Get
            Return Me.numInterval.Value
        End Get
    End Property

    ''' <summary>
    ''' Flag of thread running
    ''' </summary>
    Private Property IsRunning As Boolean

    ''' <summary>
    ''' Start stress test
    ''' </summary>
    ''' <param name="sender"></param>
    ''' <param name="e"></param>
    Private Sub btnStart_Click(ByVal sender As System.Object, ByVal e As System.EventArgs)

        Dim thread = New System.Threading.Thread(New System.Threading.ParameterizedThreadStart(AddressOf Me.runStressTest))

        Dim parameters = New Object() _
        {
            Me.RepetitionCount,
            Me.IsCategoryRandom,
            Me.SelectedCategory,
            Me.IsSeverityRandom,
            Me.SelectedSeverity,
            Me.Message
        }

        Me.setProcessingState()
        thread.Start(parameters)

    End Sub

    ''' <summary>
    ''' Stop stress test
    ''' </summary>
    ''' <param name="sender"></param>
    ''' <param name="e"></param>
    Private Sub btnStop_Click(ByVal sender As System.Object, ByVal e As System.EventArgs)

        Me.IsRunning = False

    End Sub

    ''' <summary>
    ''' Thread of logging process.
    ''' </summary>
    ''' <param name="args"></param>
    Private Sub runStressTest(args As Object)

        Dim parameters = CType(args, Object())
        Dim index = 0
        Dim repetition = System.Convert.ToInt32(parameters(index).ToString())
        index += 1
        Dim israndcat = System.Convert.ToBoolean(parameters(index))
        index += 1
        Dim category = parameters(index).ToString()
        index += 1
        Dim israndsev = System.Convert.ToBoolean(parameters(index))
        index += 1
        Dim severity = parameters(index).ToString()
        index += 1
        Dim msg = parameters(index).ToString()

        Dim repcount = 1

        ' For category random.
        Dim catrand = New System.Random()
        Dim catnames = System.Enum.GetNames(GetType(DefaultCategory))

        ' For severity random.
        Dim sevrand = New System.Random()
        Dim sevnames = System.Enum.GetNames(GetType(Severity))

        ' UI thread dispatcher.
        Dim progressDispatcher = New System.Action(Of Integer, Integer)(AddressOf Me.progressDispatcher)
        Dim errorDispatcher = New System.Action(Of String)(AddressOf Me.errorDispatcher)
        Dim finishDispatcher = New System.Action(AddressOf Me.finishDispacher)

        Me.IsRunning = True
        While Me.IsRunning And repcount <= repetition

            ' Add the number of repetitions to the message.
            Dim message = System.String.Format("Rep.{0}. {1}", repcount, msg)

            ' Get call stack.
            Dim stack = New System.Diagnostics.StackFrame(1)
            Dim path = stack.GetMethod().ToString()

            ' For random.
            If israndcat Then category = catnames(catrand.Next() Mod catnames.Length)
            If israndsev Then severity = sevnames(sevrand.Next() Mod sevnames.Length)

            ' Call LoggingFoot API
            Dim result = FtsdkVBWrapper.FTCORE_SendMessage(path, category, severity, message)

            ' Error occurred
            If result = FtsdkVBWrapper.FTCORE_RESULT.FTCORE_SUCCESS Then

                ' progress update to UI
                Me.Invoke(progressDispatcher, repcount, repetition)
                repcount += 1

                ' Interval
                If Me.RepetitionInterval > 0 Then System.Threading.Thread.CurrentThread.Join(Me.RepetitionInterval)
            Else

                Dim err = System.String.Format("Erro occurred. Result: {0}", result.ToString())
                Me.Invoke(errorDispatcher, err)
                Me.IsRunning = False
                Exit While
            End If
        End While

        ' Finish
        Me.Invoke(finishDispatcher)

    End Sub

    ''' <summary>
    ''' Update progress to UI.
    ''' </summary>
    ''' <param name="repcount">current count</param>
    ''' <param name="repetition">maximum count</param>
    Private Sub progressDispatcher(repcount As Integer, repetition As Integer)

        Dim progress = CInt((repcount / repetition) * 100.0)
        Me.prgProgress.Value = progress
        Me.lblProgress.Text = System.String.Format("{0}%", progress)

    End Sub

    ''' <summary>
    ''' Update state to UI.
    ''' </summary>
    ''' <param name="err">error message</param>
    Private Sub errorDispatcher(err As String)

        Me.setIdleState()
        System.Windows.Forms.MessageBox.Show(err)

    End Sub

    ''' <summary>
    ''' Thread of logging process finished.
    ''' </summary>
    Private Sub finishDispacher()

        Me.setIdleState()

    End Sub
#End Region 'Stress test

End Class