﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace FTSDK_Wrapper
{
    /// <summary>
    /// C# wrapper for the LoggingFoot FTSDK.
    ///
    /// <para>Copyright © 2025 Quantyworks Software</para>
    /// </summary>
    public class FtsdkCSWrapper
    {
        #region Consts
        /// <summary>
        /// LoggingFoot client module name.
        /// </summary>
        private const string _DLL_NAME = "ft_cli.dll";
        #endregion Consts

        #region Enums
        /// <summary>
        /// Represents the execution result of the LoggingFoot client.
        /// </summary>
        public enum FTCORE_RESULT
        {
            FTCORE_UNKNOWN_STATE = -1,          // The state is undefined.

            FTCORE_SUCCESS = 0,                 // Operation completed successfully.


            //-- Events --//

            FTCORE_EVT_CLIENT_DETECTDISCON = 4, // Client detected a disconnection from the server.
            FTCORE_EVT_CLIENT_RECEIVED,         // Client received data from the server.


            //-- Errors --//

            FTCORE_ERR_CLIENT_NOTSERVER = 21,   // The server is not running or does not exist.
            FTCORE_ERR_CLIENT_FAILSERVER,       // Failed to start the logging server.

            FTCORE_ERR_CLIENT_NOEXIST,          // Called before the process was created.
            FTCORE_ERR_CLIENT_ALREADY,          // The process has already been started.
            FTCORE_ERR_CLIENT_PARAMETER,        // Invalid parameters were provided.
            FTCORE_ERR_CLIENT_HOSTINFO,         // Failed to retrieve host information.
            FTCORE_ERR_CLIENT_SOCKET,           // Failed to create a socket.
            FTCORE_ERR_CLIENT_REFUSED,          // The target process is not accepting connections or refused the connection.
            FTCORE_ERR_CLIENT_UNREACHED,        // The network path to the destination is unknown.
            FTCORE_ERR_CLIENT_CONNECT,          // Connection failed due to an unknown cause.
            FTCORE_ERR_CLIENT_ESTABLISH,        // An error occurred during the connection establishment process.
            FTCORE_ERR_CLIENT_MESSAGESEND,      // Failed to send the message.
            FTCORE_ERR_CLIENT_MESSAGERECV,      // Failed to receive the message.
            FTCORE_ERR_CLIENT_CONNUNKNOWN,      // Received a connection notification from an unknown server.
        }
        #endregion Enums

        #region FTCORE External API Definitions
        /// <summary>
        /// External API definitions for FTCORE.
        /// </summary>
        private class FTCORE_APIs
        {
            /// <summary>
            /// Starts the LoggingFoot client process.
            /// </summary>
            /// <param name="servername">The server name or IP address of the destination to connect to.</param>
            /// <param name="portnumber">The port number of the destination server (49152–65535).</param>
            /// <returns>
            /// <para>SUCCESS   : FTCORE_SUCCESS</para>
            /// <para>FAIL      : FTCORE_ERR_CLIENT_ALREADY</para>
            /// <para>          : FTCORE_ERR_CLIENT_SOCKET</para>
            /// <para>          : FTCORE_ERR_CLIENT_REFUSED</para>
            /// <para>          : FTCORE_ERR_CLIENT_UNREACHED</para>
            /// <para>          : FTCORE_ERR_CLIENT_CONNECT</para>
            /// <para>          : FTCORE_ERR_CLIENT_ESTABLISH</para>
            /// </returns>
            [System.Runtime.InteropServices.DllImport(_DLL_NAME, CallingConvention = System.Runtime.InteropServices.CallingConvention.StdCall)]
            internal static extern FTCORE_RESULT FTCORE_StartProcess(string servername, ushort portnumber);

            /// <summary>
            /// Starts the client process after the server process has been launched, using the specified parameters.
            /// </summary>
            /// <param name="serverpath">The full path to the destination server executable.</param>
            /// <param name="servername">The server name or IP address to connect to.</param>
            /// <param name="portnumber">The port number of the destination server (49152–65535).</param>
            /// <param name="viewstate">The window state of the server process.
            /// <para>0: Default</para>
            /// <para>1: Maximized</para>
            /// <para>2: Minimized to the taskbar</para>
            /// <para>3: Hidden in the system tray</para>
            /// <para>4: Fully hidden</para>
            /// </param>
            /// <param name="topmost">
            /// Specifies whether the window should stay on top of other windows.
            /// <para>0: Disabled</para>
            /// <para>1: Enabled</para>
            /// </param>
            /// <returns>
            /// <para>SUCCESS   : FTCORE_SUCCESS</para>
            /// <para>FAIL      : FTCORE_ERR_CLIENT_NOTSERVER</para>
            /// <para>          : FTCORE_ERR_CLIENT_FAILSERVER</para>
            /// <para>          : FTCORE_ERR_CLIENT_ALREADY</para>
            /// <para>          : FTCORE_ERR_CLIENT_SOCKET</para>
            /// <para>          : FTCORE_ERR_CLIENT_REFUSED</para>
            /// <para>          : FTCORE_ERR_CLIENT_UNREACHED</para>
            /// <para>          : FTCORE_ERR_CLIENT_CONNECT</para>
            /// <para>          : FTCORE_ERR_CLIENT_ESTABLISH</para>
            /// </returns>
            [System.Runtime.InteropServices.DllImport(_DLL_NAME, CallingConvention = System.Runtime.InteropServices.CallingConvention.StdCall)]
            internal static extern FTCORE_RESULT FTCORE_StartTriggerWithParam(string serverpath, string servername, ushort portnumber, int viewstate, int topmost);

            /// <summary>
            /// Starts the client process after the server process has been launched, using the setup parameters.
            /// </summary>
            /// <param name="serverpath">The full path to the destination server executable.</param>
            /// <param name="servername">The server name or IP address to connect to.</param>
            /// <param name="portnumber">The port number of the destination server (49152–65535).</param>
            /// <returns>
            /// <para>SUCCESS   : FTCORE_SUCCESS</para>
            /// <para>FAIL      : FTCORE_ERR_CLIENT_NOTSERVER</para>
            /// <para>          : FTCORE_ERR_CLIENT_FAILSERVER</para>
            /// <para>          : FTCORE_ERR_CLIENT_ALREADY</para>
            /// <para>          : FTCORE_ERR_CLIENT_SOCKET</para>
            /// <para>          : FTCORE_ERR_CLIENT_REFUSED</para>
            /// <para>          : FTCORE_ERR_CLIENT_UNREACHED</para>
            /// <para>          : FTCORE_ERR_CLIENT_CONNECT</para>
            /// <para>          : FTCORE_ERR_CLIENT_ESTABLISH</para>
            /// </returns>
            [System.Runtime.InteropServices.DllImport(_DLL_NAME, CallingConvention = System.Runtime.InteropServices.CallingConvention.StdCall)]
            internal static extern FTCORE_RESULT FTCORE_StartTriggerWithSetup(string serverpath, string servername, ushort portnumber);

            /// <summary>
            /// Terminates the client process.
            /// This function must be called to terminate the process if it was started using FTCORE_StartProcess().
            /// </summary>
            /// <returns>
            /// <para>SUCCESS   : FTCORE_SUCCESS</para>
            /// <para>FAIL      : -----</para>
            /// </returns>
            [System.Runtime.InteropServices.DllImport(_DLL_NAME, CallingConvention = System.Runtime.InteropServices.CallingConvention.StdCall)]
            internal static extern FTCORE_RESULT FTCORE_ExitProcess();

            /// <summary>
            /// Terminates the client process after the server process has exited.
            /// This function must be called to terminate the process if it was started using either FTCORE_StartTriggerWithParam() or FTCORE_StartTriggerWithSetup().
            /// If no server process is running, only the client process will be terminated.
            /// </summary>
            /// <returns>
            /// <para>SUCCESS   : FTCORE_SUCCESS</para>
            /// <para>FAIL      : -----</para>
            /// </returns>
            [System.Runtime.InteropServices.DllImport(_DLL_NAME, CallingConvention = System.Runtime.InteropServices.CallingConvention.StdCall)]
            internal static extern FTCORE_RESULT FTCORE_ExitTrigger();

            /// <summary>
            /// Sends a message to the server process.
            /// </summary>
            /// <param name="path">The caller's path or source identifier.</param>
            /// <param name="category">The log category.</param>
            /// <param name="severity">The log severity level.</param>
            /// <param name="message">The message content to send.</param>
            /// <returns>
            /// <para>SUCCESS   : FTCORE_SUCCESS</para>
            /// <para>FAIL      : FTCORE_ERR_CLIENT_NOEXIST</para>
            /// <para>          : FTCORE_ERR_CLIENT_PARAMETER</para>
            /// <para>          : FTCORE_ERR_CLIENT_MESSAGESEND</para>
            /// </returns>
            [System.Runtime.InteropServices.DllImport(_DLL_NAME, CallingConvention = System.Runtime.InteropServices.CallingConvention.StdCall)]
            internal static extern FTCORE_RESULT FTCORE_SendMessage(string path, string category, string severity, string message);

            /// <summary>
            /// Instructs the server process to change its window display state.
            /// </summary>
            /// <param name="state">
            /// The desired window state of the server process.
            /// <para>0: Default</para>
            /// <para>1: Maximized</para>
            /// <para>2: Minimized to the taskbar</para>
            /// <para>3: Hidden in the system tray</para>
            /// <para>4: Fully hidden</para>
            /// </param>
            /// <returns>
            /// <para>SUCCESS   : FTCORE_SUCCESS</para>
            /// <para>FAIL      : FTCORE_ERR_CLIENT_NOEXIST</para>
            /// <para>          : FTCORE_ERR_CLIENT_PARAMETER</para>
            /// <para>          : FTCORE_ERR_CLIENT_MESSAGESEND</para>
            /// </returns>
            [System.Runtime.InteropServices.DllImport(_DLL_NAME, CallingConvention = System.Runtime.InteropServices.CallingConvention.StdCall)]
            internal static extern FTCORE_RESULT FTCORE_SetWindowState(int state);

            /// <summary>
            /// Instructs the server process to load the default logging category values.
            /// </summary>
            /// <returns>
            /// <para>SUCCESS   : FTCORE_SUCCESS</para>
            /// <para>FAIL      : FTCORE_ERR_CLIENT_NOEXIST</para>
            /// <para>          : FTCORE_ERR_CLIENT_PARAMETER</para>
            /// <para>          : FTCORE_ERR_CLIENT_MESSAGESEND</para>
            /// </returns>
            [System.Runtime.InteropServices.DllImport(_DLL_NAME, CallingConvention = System.Runtime.InteropServices.CallingConvention.StdCall)]
            internal static extern FTCORE_RESULT FTCORE_LoadDefaultCategory();

            /// <summary>
            /// Instructs the server process to apply custom logging categories.
            /// </summary>
            /// <param name="categories">One or more category names, separated by commas if multiple.</param>
            /// <returns>
            /// <para>SUCCESS   : FTCORE_SUCCESS</para>
            /// <para>FAIL      : FTCORE_ERR_CLIENT_NOEXIST</para>
            /// <para>          : FTCORE_ERR_CLIENT_PARAMETER</para>
            /// <para>          : FTCORE_ERR_CLIENT_MESSAGESEND</para>
            /// </returns>
            [System.Runtime.InteropServices.DllImport(_DLL_NAME, CallingConvention = System.Runtime.InteropServices.CallingConvention.StdCall)]
            internal static extern FTCORE_RESULT FTCORE_SetCustomCategory(string categories);

            /// <summary>
            /// Instructs the server process to apply the loaded category settings from the specified file.
            /// </summary>
            /// <param name="filename">The full path to the category settings file.</param>
            /// <returns>
            /// <para>SUCCESS   : FTCORE_SUCCESS</para>
            /// <para>FAIL      : FTCORE_ERR_CLIENT_NOEXIST</para>
            /// <para>          : FTCORE_ERR_CLIENT_PARAMETER</para>
            /// <para>          : FTCORE_ERR_CLIENT_MESSAGESEND</para>
            /// </returns>
            [System.Runtime.InteropServices.DllImport(_DLL_NAME, CallingConvention = System.Runtime.InteropServices.CallingConvention.StdCall)]
            internal static extern FTCORE_RESULT FTCORE_SetCustomCategoryFromFile(string filename);
        }
        #endregion FTCORE External API Definitions

        #region LoggingClient Wrapper
        /// <summary>
        /// Starts the LoggingFoot client process.
        /// </summary>
        /// <param name="servername">The server name or IP address of the destination to connect to.</param>
        /// <param name="portnumber">The port number of the destination server (49152–65535).</param>
        /// <returns>
        /// <para>SUCCESS   : FTCORE_SUCCESS</para>
        /// <para>FAIL      : FTCORE_ERR_CLIENT_ALREADY</para>
        /// <para>          : FTCORE_ERR_CLIENT_SOCKET</para>
        /// <para>          : FTCORE_ERR_CLIENT_REFUSED</para>
        /// <para>          : FTCORE_ERR_CLIENT_UNREACHED</para>
        /// <para>          : FTCORE_ERR_CLIENT_CONNECT</para>
        /// <para>          : FTCORE_ERR_CLIENT_ESTABLISH</para>
        /// </returns>
        public static FTCORE_RESULT FTCORE_StartProcess(string servername, int portnumber)
        {
            return FTCORE_APIs.FTCORE_StartProcess(servername, (ushort)portnumber);
        }

        /// <summary>
        /// After the server process is started with the parameters passed, the client process is started.
        /// </summary>
        /// <param name="serverpath">The full path to the destination server executable.</param>
        /// <param name="servername">The server name or IP address to connect to.</param>
        /// <param name="portnumber">The port number of the destination server (49152–65535).</param>
        /// <param name="viewstate">The window state of the server process.
        /// <para>0: Default</para>
        /// <para>1: Maximized</para>
        /// <para>2: Minimized to the taskbar</para>
        /// <para>3: Hidden in the system tray</para>
        /// <para>4: Fully hidden</para>
        /// </param>
        /// <param name="topmost">
        /// Specifies whether the window should stay on top of other windows.
        /// <para>0: Disabled</para>
        /// <para>1: Enabled</para>
        /// </param>
        /// <returns>
        /// <para>SUCCESS   : FTCORE_SUCCESS</para>
        /// <para>FAIL      : FTCORE_ERR_CLIENT_NOTSERVER</para>
        /// <para>          : FTCORE_ERR_CLIENT_FAILSERVER</para>
        /// <para>          : FTCORE_ERR_CLIENT_ALREADY</para>
        /// <para>          : FTCORE_ERR_CLIENT_SOCKET</para>
        /// <para>          : FTCORE_ERR_CLIENT_REFUSED</para>
        /// <para>          : FTCORE_ERR_CLIENT_UNREACHED</para>
        /// <para>          : FTCORE_ERR_CLIENT_CONNECT</para>
        /// <para>          : FTCORE_ERR_CLIENT_ESTABLISH</para>
        /// </returns>
        public static FTCORE_RESULT FTCORE_StartTriggerWithParam(string serverpath, string servername, int portnumber, int viewstate, bool topmost)
        {
            var istop = topmost ? 1 : 0;
            return FTCORE_APIs.FTCORE_StartTriggerWithParam(serverpath, servername, (ushort)portnumber, viewstate, istop);
        }

        /// <summary>
        /// Starts the client process after the server process has been launched, using the setup parameters.
        /// </summary>
        /// <param name="serverpath">The full path to the destination server executable.</param>
        /// <param name="servername">The server name or IP address to connect to.</param>
        /// <param name="portnumber">The port number of the destination server (49152–65535).</param>
        /// <returns>
        /// <para>SUCCESS   : FTCORE_SUCCESS</para>
        /// <para>FAIL      : FTCORE_ERR_CLIENT_NOTSERVER</para>
        /// <para>          : FTCORE_ERR_CLIENT_FAILSERVER</para>
        /// <para>          : FTCORE_ERR_CLIENT_ALREADY</para>
        /// <para>          : FTCORE_ERR_CLIENT_SOCKET</para>
        /// <para>          : FTCORE_ERR_CLIENT_REFUSED</para>
        /// <para>          : FTCORE_ERR_CLIENT_UNREACHED</para>
        /// <para>          : FTCORE_ERR_CLIENT_CONNECT</para>
        /// <para>          : FTCORE_ERR_CLIENT_ESTABLISH</para>
        /// </returns>
        public static FTCORE_RESULT FTCORE_StartTriggerWithSetup(string serverpath, string servername, int portnumber)
        {
            return FTCORE_APIs.FTCORE_StartTriggerWithSetup(serverpath, servername, (ushort)portnumber);
        }

        /// <summary>
        /// Terminates the client process.
        /// This function must be called to terminate the process if it was started using FTCORE_StartProcess().
        /// </summary>
        /// <returns>
        /// <para>SUCCESS   : FTCORE_SUCCESS</para>
        /// <para>FAIL      : -----</para>
        /// </returns>
        public static FTCORE_RESULT FTCORE_ExitProcess()
        {
            return FTCORE_APIs.FTCORE_ExitProcess();
        }

        /// <summary>
        /// Terminates the client process after the server process has exited.
        /// This function must be called to terminate the process if it was started using either FTCORE_StartTriggerWithParam() or FTCORE_StartTriggerWithSetup().
        /// If no server process is running, only the client process will be terminated.
        /// </summary>
        /// <returns>
        /// <para>SUCCESS   : FTCORE_SUCCESS</para>
        /// <para>FAIL      : -----</para>
        /// </returns>
        public static FTCORE_RESULT FTCORE_ExitTrigger()
        {
            return FTCORE_APIs.FTCORE_ExitTrigger();
        }

        /// <summary>
        /// Sends a message to the server process.
        /// </summary>
        /// <param name="path">The caller's path or source identifier.</param>
        /// <param name="category">The log category.</param>
        /// <param name="severity">The log severity level.</param>
        /// <param name="message">The message content to send.</param>
        /// <returns>
        /// <para>SUCCESS   : FTCORE_SUCCESS</para>
        /// <para>FAIL      : FTCORE_ERR_CLIENT_NOEXIST</para>
        /// <para>          : FTCORE_ERR_CLIENT_PARAMETER</para>
        /// <para>          : FTCORE_ERR_CLIENT_MESSAGESEND</para>
        /// </returns>
        public static FTCORE_RESULT FTCORE_SendMessage(string path, string category, string severity, string message)
        {
            return FTCORE_APIs.FTCORE_SendMessage(path, category, severity, message);
        }

        /// <summary>
        /// Instructs the server process to change its window display state.
        /// </summary>
        /// <param name="state">
        /// The desired window state of the server process.
        /// <para>0: Default</para>
        /// <para>1: Maximized</para>
        /// <para>2: Minimized to the taskbar</para>
        /// <para>3: Hidden in the system tray</para>
        /// <para>4: Fully hidden</para>
        /// </param>
        /// <returns>
        /// <para>SUCCESS   : FTCORE_SUCCESS</para>
        /// <para>FAIL      : FTCORE_ERR_CLIENT_NOEXIST</para>
        /// <para>          : FTCORE_ERR_CLIENT_PARAMETER</para>
        /// <para>          : FTCORE_ERR_CLIENT_MESSAGESEND</para>
        /// </returns>
        public static FTCORE_RESULT FTCORE_SetWindowState(int state)
        {
            return FTCORE_APIs.FTCORE_SetWindowState(state);
        }

        /// <summary>
        /// Instructs the server process to load the default logging category values.
        /// </summary>
        /// <returns>
        /// <para>SUCCESS   : FTCORE_SUCCESS</para>
        /// <para>FAIL      : FTCORE_ERR_CLIENT_NOEXIST</para>
        /// <para>          : FTCORE_ERR_CLIENT_PARAMETER</para>
        /// <para>          : FTCORE_ERR_CLIENT_MESSAGESEND</para>
        /// </returns>
        public static FTCORE_RESULT FTCORE_LoadDefaultCategory()
        {
            return FTCORE_APIs.FTCORE_LoadDefaultCategory();
        }

        /// <summary>
        /// Instructs the server process to apply custom logging categories.
        /// </summary>
        /// <param name="categories">One or more category names, separated by commas if multiple.</param>
        /// <returns>
        /// <para>SUCCESS   : FTCORE_SUCCESS</para>
        /// <para>FAIL      : FTCORE_ERR_CLIENT_NOEXIST</para>
        /// <para>          : FTCORE_ERR_CLIENT_PARAMETER</para>
        /// <para>          : FTCORE_ERR_CLIENT_MESSAGESEND</para>
        /// </returns>
        public static FTCORE_RESULT FTCORE_SetCustomCategory(string categories)
        {
            return FTCORE_APIs.FTCORE_SetCustomCategory(categories);
        }

        /// <summary>
        /// Instructs the server process to apply the loaded category settings from the specified file.
        /// </summary>
        /// <param name="filename">Full path of category setting file name.</param>
        /// <returns>
        /// <para>SUCCESS   : FTCORE_SUCCESS</para>
        /// <para>FAIL      : FTCORE_ERR_CLIENT_NOEXIST</para>
        /// <para>          : FTCORE_ERR_CLIENT_PARAMETER</para>
        /// <para>          : FTCORE_ERR_CLIENT_MESSAGESEND</para>
        /// </returns>
        public static FTCORE_RESULT FTCORE_SetCustomCategoryFromFile(string filename)
        {
            return FTCORE_APIs.FTCORE_SetCustomCategoryFromFile(filename);
        }
        #endregion LoggingClient Wrapper
    }
}
